/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.parser.html;

import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;

import junit.framework.TestCase;

public class HTMLParserTest extends TestCase {

	static class Hn implements HTMLHandler {

		StringBuffer buf = new StringBuffer();

		@Override
		public void string(String s) {
			buf.append(s).append('\n');
		}

		@Override
		public void startTag(String s) {
			buf.append("tag name:").append(s).append('\n');
		}

		@Override
		public void endTag(String s) {
			buf.append("tag end:").append(s).append('\n');
		}

		@Override
		public void tagAttribute(String k, String v) {
			buf.append("tag attribute:(").append(k).append(',')
			.append(v).append(")\n");
		}

		@Override
		public void meta(String s) throws SQLException {
			// do nothing
		}

	};

	private void eqs(String s, String t) {
		StringReader r = new StringReader(s);
		Hn h = new Hn();

		try {
			HTMLParser.parse(h, r);
			assertEquals(h.buf.toString(), t);
		} catch(IOException e) {
			throw new RuntimeException(e);
		} catch(SQLException e) {
			throw new RuntimeException(e);
		}
	}

	public void testHTML001() {
		eqs("aaaa", "aaaa\n");
	}

	public void testHTML002() {
		eqs("<aaaa>", "tag name:aaaa\n");
		eqs("aaa<aaaa>", "aaa\ntag name:aaaa\n");
		eqs("<aaaa>aaa", "tag name:aaaa\naaa\n");
		eqs("<   aaaa>", "tag name:aaaa\n");
		eqs("<aaaa   >", "tag name:aaaa\n");
	}

	public void testHTML003() {
		eqs("<aaaa  bbb>", "tag name:aaaa\ntag attribute:(bbb,)\n");
		eqs("aaa<aaaa  bbb>aaa", "aaa\ntag name:aaaa\ntag attribute:(bbb,)\naaa\n");
		eqs("<aaaa  bbb   >", "tag name:aaaa\ntag attribute:(bbb,)\n");
	}

	public void testHTML004() {
		eqs("<aaaa  bbb=ccc>",
				"tag name:aaaa\ntag attribute:(bbb,ccc)\n");
		eqs("aaa<aaaa bbb=ccc>aaa",
				"aaa\ntag name:aaaa\ntag attribute:(bbb,ccc)\naaa\n");
		eqs("<aaaa  bbb=ccc    >",
				"tag name:aaaa\ntag attribute:(bbb,ccc)\n");
	}

	public void testHTML005() {
		eqs("<aaaa bbb=\"ccc\">",
				"tag name:aaaa\ntag attribute:(bbb,ccc)\n");
		eqs("aaa<aaaa bbb=\"ccc\">aaa",
				"aaa\ntag name:aaaa\ntag attribute:(bbb,ccc)\naaa\n");
		eqs("aaa<aaaa bbb=\"c'cc\">aaa",
				"aaa\ntag name:aaaa\ntag attribute:(bbb,c'cc)\naaa\n");
		eqs("<aaaa bbb=\"ccc\"   >",
				"tag name:aaaa\ntag attribute:(bbb,ccc)\n");
	}

	public void testHTML006() {
		eqs("<aaaa bbb='ccc'>",
				"tag name:aaaa\ntag attribute:(bbb,ccc)\n");
		eqs("aaa<aaaa bbb='ccc'>aaa",
				"aaa\ntag name:aaaa\ntag attribute:(bbb,ccc)\naaa\n");
		eqs("aaa<aaaa bbb='c\"cc'>aaa",
				"aaa\ntag name:aaaa\ntag attribute:(bbb,c\"cc)\naaa\n");
		eqs("<aaaa bbb='ccc'   >",
				"tag name:aaaa\ntag attribute:(bbb,ccc)\n");
	}

	public void testHTML007() {
		eqs("<aaaa bbb=ccc ddd=eee>",
				"tag name:aaaa\n"
				+ "tag attribute:(bbb,ccc)\n"
				+ "tag attribute:(ddd,eee)\n");
		eqs("aaa<aaaa bbb=ccc ddd=eee>aaa",
				"aaa\n"
				+ "tag name:aaaa\n"
				+ "tag attribute:(bbb,ccc)\n"
				+ "tag attribute:(ddd,eee)\n"
				+ "aaa\n");
		eqs("<aaaa bbb=ccc ddd=eee   >",
				"tag name:aaaa\n"
				+ "tag attribute:(bbb,ccc)\n"
				+ "tag attribute:(ddd,eee)\n");
	}

	public void testHTML008() {
		eqs("<aaaa bbb=ccc ddd=\"eee\">",
				"tag name:aaaa\n"
				+ "tag attribute:(bbb,ccc)\n"
				+ "tag attribute:(ddd,eee)\n");
		eqs("aaa<aaaa bbb=ccc ddd=\"eee\">aaa",
				"aaa\n"
				+ "tag name:aaaa\n"
				+ "tag attribute:(bbb,ccc)\n"
				+ "tag attribute:(ddd,eee)\n"
				+ "aaa\n");
		eqs("<aaaa bbb=ccc ddd='eee'>",
				"tag name:aaaa\n"
				+ "tag attribute:(bbb,ccc)\n"
				+ "tag attribute:(ddd,eee)\n");
		eqs("aaa<aaaa bbb=ccc ddd='eee'>aaa",
				"aaa\n"
				+ "tag name:aaaa\n"
				+ "tag attribute:(bbb,ccc)\n"
				+ "tag attribute:(ddd,eee)\n"
				+ "aaa\n");
		eqs("<aaaa bbb=ccc ddd=\"eee\"   >",
				"tag name:aaaa\n"
				+ "tag attribute:(bbb,ccc)\n"
				+ "tag attribute:(ddd,eee)\n");
	}

	public void testHTML009() {
		eqs("</aaaa>", "tag end:aaaa\n");
		eqs("aaa</aaaa>aaa", "aaa\ntag end:aaaa\naaa\n");
		eqs("<   /aaaa>", "tag end:aaaa\n");
		eqs("</   aaaa>", "tag end:aaaa\n");
		eqs("</aaaa   >", "tag end:aaaa\n");
	}

	public void testHTML010() {
		eqs("<aaaa/>", "tag name:aaaa\ntag end:aaaa\n");
		eqs("<aaaa />", "tag name:aaaa\ntag end:aaaa\n");
		eqs("<   aaaa />", "tag name:aaaa\ntag end:aaaa\n");
		eqs("<aaaa /   >", "tag name:aaaa\ntag end:aaaa\n");
	}

	public void testHTML011() {
		eqs("&amp;", "&\n");
		eqs("&amp;a", "&a\n");
		eqs("&amp", "&\n");
		eqs("&amp  a", "&  a\n");
	}

	public void testHTML012() {
		eqs("&#38;", "&\n");
		eqs("&#38", "&\n");
		eqs("&#38  a", "&  a\n");
		eqs("&#00000000038;", "&\n");
	}

	public void testHTML101() {
		eqs("<aaa>aaaa</aaa>",
				"tag name:aaa\n"
				+ "aaaa\n"
				+ "tag end:aaa\n");
	}

}
