/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.relations;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.morilib.db.engine.SqlEngine;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.dml.SqlSelectDistinct;

public abstract class AbstractRelation implements Relation {

	public Relation project(
			final List<String> names) throws SQLException {
		final RelationCursor t = iterator();

		if(!getColumnNames().containsAll(names)) {
			throw ErrorBundle.getDefault(10004);
		}

		return new AbstractRelation() {

			public RelationCursor iterator() {
				return new RelationCursor() {

					public boolean hasNext() {
						return t.hasNext();
					}

					public RelationTuple next(
							) throws IOException, SQLException {
						final RelationTuple x = t.next();

						return new RelationTuple() {

							public Object get(
									String name) throws SQLException {
								if(!names.contains(name)) {
									throw ErrorBundle.getDefault(10009, name);
								}
								return x.get(name);
							}

							@Override
							public RelationTuple copy() throws SQLException {
								return new DefaultRelationTuple(toMap());
							}

							@Override
							public Map<String, Object> toMap() throws SQLException {
								List<SqlColumnDefinition> ases = getColumnNames();
								Map<String, Object> m;

								m = new LinkedHashMap<String, Object>();
								for(int i = 0; i < ases.size(); i++) {
									m.put(ases.get(i).getName(),
											get(ases.get(i).getName()));
								}
								return m;
							}

						};
					}

				};
			}

			public List<SqlColumnDefinition> getColumnNames() {
				List<SqlColumnDefinition> l;

				l = new ArrayList<SqlColumnDefinition>();
				for(String s : names)  l.add(getDefinition(s));
				return l;
			}

			@Override
			public SqlColumnDefinition getDefinition(String name) {
				return AbstractRelation.this.getDefinition(name);
			}

		};
	}

	public Relation select(SqlEngine v, SqlSchema f,
			RelationExpression pred, RelationAggregate m,
			SqlSelectDistinct d, RelationTuple c,
			List<String> group,
			List<Object> h) throws IOException, SQLException {
		final Collection<RelationTuple> r = d.create();
		RelationCursor i = iterator();
		RelationTuple x, t;

		while(i.hasNext()) {
			x = i.next().copy();
			t = new CorrelatedRelationTuple(x, c);
			if(pred.test(v, f, t, m, group, h).isTrue()) {
				r.add(x);
			}
		}
		return new TableRelation(getColumnNames(), r);
	}

}
