/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.sql;

import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;

import junit.framework.TestCase;

public class DbSqlSelectTest extends TestCase {

	void okp(String s) {
		DbSqlLexer l;

		try {
			l = new DbSqlLexer(new StringReader(s));
			new DbSqlParser()._top(l);
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	void eqs(String s, String t) {
		DbSqlLexer l;

		try {
			l = new DbSqlLexer(new StringReader(s));
			l.eatsym("SELECT");
			assertEquals(t, new DbSqlParser()._select(l, null).toString());
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	public void testSelectParse001() {
		okp("SELECT 1");
		okp("SELECT 1 AS a");
		okp("SELECT 1 a");
		okp("SELECT 1, 2");
		okp("SELECT 1 * 2");
		okp("SELECT 1 FROM dual");
		okp("SELECT 1 FROM dual, duo");
		okp("SELECT 1 WHERE a = 1");
		okp("SELECT 1 FROM dual WHERE a = 1");
		okp("SELECT 1 FROM dual GROUP BY a");
		okp("SELECT 1 FROM dual WHERE A = 1 GROUP BY a");
		okp("SELECT 1 FROM dual WHERE A = 1 GROUP BY a, b");
		okp("SELECT 1 FROM dual GROUP BY a HAVING a = 1");
		okp("SELECT 1 FROM dual WHERE A = 1 GROUP BY a HAVING a = 1");
		okp("SELECT 1 FROM dual WHERE A = 1 GROUP BY a, b HAVING a = 1");
		okp("SELECT 1 FROM dual ORDER BY a");
		okp("SELECT 1 FROM dual ORDER BY a, b");
		okp("SELECT 1 FROM dual WHERE A = 1 ORDER BY a, b");
		okp("SELECT 1 FROM dual GROUP BY a ORDER BY a, b");
		okp("SELECT 1 FROM dual WHERE A = 1 GROUP BY a ORDER BY a, b");
		okp("SELECT 1 FROM dual GROUP BY a HAVING a = 1 ORDER BY a, b");
		okp("SELECT 1 FROM dual WHERE A = 1 GROUP BY a HAVING a = 1 ORDER BY a, b");
	}

	public void testSelectParse002() {
		okp("SELECT 1 FROM b JOIN c");
		okp("SELECT 1 FROM b x JOIN c y");
		okp("SELECT 1 FROM b AS x JOIN c AS y");
		okp("SELECT 1 FROM b JOIN c ON a = 1");
		okp("SELECT 1 FROM b AS x JOIN c AS y ON a = 1");
		okp("SELECT 1 FROM b JOIN c ON a = 1, d");
		okp("SELECT 1 FROM b JOIN c ON a = 1 JOIN d");
		okp("SELECT 1 FROM b JOIN c ON a = 1 JOIN d ON b = 1");
		okp("SELECT 1 FROM b INNER JOIN c ON a = 1");
		okp("SELECT 1 FROM b LEFT JOIN c ON a = 1");
		okp("SELECT 1 FROM b LEFT OUTER JOIN c ON a = 1");
		okp("SELECT 1 FROM b RIGHT JOIN c ON a = 1");
		okp("SELECT 1 FROM b RIGHT OUTER JOIN c ON a = 1");
		okp("SELECT 1 FROM b FULL JOIN c ON a = 1");
		okp("SELECT 1 FROM b FULL OUTER JOIN c ON a = 1");
		okp("SELECT 1 FROM b OUTER JOIN c ON a = 1");
		okp("SELECT 1 FROM b NATURAL JOIN c");
		okp("SELECT 1 FROM b CROSS JOIN c");
	}

	public void testSelectParse003() {
		okp("SELECT 1 FROM (SELECT 2 AS a)");
		okp("SELECT 1 FROM (SELECT 2 AS a) b");
		okp("SELECT 1 FROM (SELECT 2 AS a) AS b");
	}

	public void testSelectParse004() {
		okp("SELECT EXISTS (SELECT 1 FROM b)");
		okp("SELECT (SELECT 1 FROM b) a FROM b");
		okp("SELECT 1 FROM b WHERE a = (SELECT 1 FROM b)");
	}

	public void testSelect001() {
		eqs("SELECT 1", "(select ((1 . #f)) #f #t #f #t #f)");
		eqs("SELECT 1 AS a", "(select ((1 . A)) #f #t #f #t #f)");
		eqs("SELECT 1 a", "(select ((1 . A)) #f #t #f #t #f)");
		eqs("SELECT 1, 2", "(select ((1 . #f) (2 . #f)) #f #t #f #t #f)");
		eqs("SELECT 1 * 2", "(select (((* 1 2) . #f)) #f #t #f #t #f)");
		eqs("SELECT 1 FROM dual", "(select ((1 . #f)) (DUAL) #t #f #t #f)");
		eqs("SELECT 1 FROM dual, duo", "(select ((1 . #f)) (DUAL DUO) #t #f #t #f)");
		eqs("SELECT 1 WHERE a = 1", "(select ((1 . #f)) #f (= A 1) #f #t #f)");
		eqs("SELECT 1 FROM dual WHERE a = 1", "(select ((1 . #f)) (DUAL) (= A 1) #f #t #f)");
		eqs("SELECT 1 FROM dual GROUP BY a", "(select ((1 . #f)) (DUAL) #t (A) #t #f)");
		eqs("SELECT 1 FROM dual WHERE A = 1 GROUP BY a",
				"(select ((1 . #f)) (DUAL) (= A 1) (A) #t #f)");
		eqs("SELECT 1 FROM dual WHERE A = 1 GROUP BY a, b",
				"(select ((1 . #f)) (DUAL) (= A 1) (A B) #t #f)");
		eqs("SELECT 1 FROM dual GROUP BY a HAVING a = 1",
				"(select ((1 . #f)) (DUAL) #t (A) (= A 1) #f)");
		eqs("SELECT 1 FROM dual WHERE A = 1 GROUP BY a HAVING a = 1",
				"(select ((1 . #f)) (DUAL) (= A 1) (A) (= A 1) #f)");
		eqs("SELECT 1 FROM dual WHERE A = 1 GROUP BY a, b HAVING a = 1",
				"(select ((1 . #f)) (DUAL) (= A 1) (A B) (= A 1) #f)");
		eqs("SELECT 1 FROM dual ORDER BY a", "(select ((1 . #f)) (DUAL) #t #f #t ((A . ASC)))");
		eqs("SELECT 1 FROM dual ORDER BY a, b",
				"(select ((1 . #f)) (DUAL) #t #f #t ((A . ASC) (B . ASC)))");
		eqs("SELECT 1 FROM dual WHERE A = 1 ORDER BY a, b",
				"(select ((1 . #f)) (DUAL) (= A 1) #f #t ((A . ASC) (B . ASC)))");
		eqs("SELECT 1 FROM dual GROUP BY a ORDER BY a, b",
				"(select ((1 . #f)) (DUAL) #t (A) #t ((A . ASC) (B . ASC)))");
		eqs("SELECT 1 FROM dual WHERE A = 1 GROUP BY a ORDER BY a, b",
				"(select ((1 . #f)) (DUAL) (= A 1) (A) #t ((A . ASC) (B . ASC)))");
		eqs("SELECT 1 FROM dual GROUP BY a HAVING a = 1 ORDER BY a, b",
				"(select ((1 . #f)) (DUAL) #t (A) (= A 1) ((A . ASC) (B . ASC)))");
		eqs("SELECT 1 FROM dual WHERE A = 1 GROUP BY a HAVING a = 1 ORDER BY a, b",
				"(select ((1 . #f)) (DUAL) (= A 1) (A) (= A 1) ((A . ASC) (B . ASC)))");
	}

	public void testSelect002() {
		eqs("SELECT 1 FROM b JOIN c", "(select ((1 . #f)) ((join inner B C)) #t #f #t #f)");
		eqs("SELECT 1 FROM b x JOIN c y",
				"(select ((1 . #f)) ((join inner (as B X) (as C Y))) #t #f #t #f)");
		eqs("SELECT 1 FROM b AS x JOIN c AS y",
				"(select ((1 . #f)) ((join inner (as B X) (as C Y))) #t #f #t #f)");
		eqs("SELECT 1 FROM b JOIN c ON a = 1",
				"(select ((1 . #f)) ((join inner B C (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b AS x JOIN c AS y ON a = 1",
				"(select ((1 . #f)) ((join inner (as B X) (as C Y) (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b JOIN c ON a = 1, d",
				"(select ((1 . #f)) ((join inner B C (= A 1)) D) #t #f #t #f)");
		eqs("SELECT 1 FROM b JOIN c ON a = 1 JOIN d",
				"(select ((1 . #f)) ((join inner (join inner B C (= A 1)) D)) #t #f #t #f)");
		eqs("SELECT 1 FROM b JOIN c ON a = 1 JOIN d ON b = 1",
				"(select ((1 . #f)) ((join inner (join inner B C (= A 1)) D (= B 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b INNER JOIN c ON a = 1",
				"(select ((1 . #f)) ((join inner B C (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b LEFT JOIN c ON a = 1",
				"(select ((1 . #f)) ((join left-outer B C (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b LEFT OUTER JOIN c ON a = 1",
				"(select ((1 . #f)) ((join left-outer B C (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b RIGHT JOIN c ON a = 1",
				"(select ((1 . #f)) ((join right-outer B C (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b RIGHT OUTER JOIN c ON a = 1",
				"(select ((1 . #f)) ((join right-outer B C (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b FULL JOIN c ON a = 1",
				"(select ((1 . #f)) ((join full-outer B C (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b FULL OUTER JOIN c ON a = 1",
				"(select ((1 . #f)) ((join full-outer B C (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b OUTER JOIN c ON a = 1",
				"(select ((1 . #f)) ((join left-outer B C (= A 1))) #t #f #t #f)");
		eqs("SELECT 1 FROM b NATURAL JOIN c",
				"(select ((1 . #f)) ((join natural B C)) #t #f #t #f)");
		eqs("SELECT 1 FROM b CROSS JOIN c",
				"(select ((1 . #f)) ((join cross B C)) #t #f #t #f)");
	}

	public void testSelect003() {
		eqs("SELECT 1 FROM (SELECT 2 AS a)",
				"(select ((1 . #f))" +
				" ((subrelation (select ((2 . A)) #f #t #f #t #f))) #t #f #t #f)");
		eqs("SELECT 1 FROM (SELECT 2 AS a) b",
				"(select ((1 . #f))" +
				" ((as (subrelation (select ((2 . A)) #f #t #f #t #f)) B)) #t #f #t #f)");
		eqs("SELECT 1 FROM (SELECT 2 AS a) AS b",
				"(select ((1 . #f))" +
				" ((as (subrelation (select ((2 . A)) #f #t #f #t #f)) B)) #t #f #t #f)");
	}

	public void testSelect004() {
		eqs("SELECT EXISTS (SELECT 1 FROM b)",
				"(select (((exists (select ((1 . #f)) (B) #t #f #t #f)) . #f)) #f #t #f #t #f)");
		eqs("SELECT (SELECT 1 FROM b) a FROM b",
				"(select (((subquery-literal (select ((1 . #f)) (B) #t #f #t #f)) . A))" +
				" (B) #t #f #t #f)");
		eqs("SELECT 1 FROM b WHERE a = (SELECT 1 FROM b)",
				"(select ((1 . #f)) (B)" +
				" (= A (subquery-literal (select ((1 . #f)) (B) #t #f #t #f))) #f #t #f)");
	}

}
