/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.sql;

import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;

import junit.framework.TestCase;

public class DbSqlSetParserTest extends TestCase {

	void okp(String s) {
		DbSqlLexer l;

		try {
			l = new DbSqlLexer(new StringReader(s));
			new DbSqlSetParser().parse(l);
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	void eqp(String s, String t) {
		DbSqlLexer l;

		try {
			l = new DbSqlLexer(new StringReader(s));
			assertEquals(t, new DbSqlSetParser().parse(l).toString());
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	public void testParser001() {
		okp("SELECT 1");
	}

	public void testParser002() {
		okp("SELECT 1 UNION SELECT 2");
		okp("SELECT 1 UNION ALL SELECT 2");
		okp("SELECT 1 MINUS SELECT 2");
		okp("SELECT 1 EXCEPT SELECT 2");
		okp("SELECT 1 INTERSECT SELECT 2");
		okp("SELECT 1 DIVIDE SELECT 2");
	}

	public void testParser003() {
		okp("SELECT 1 UNION SELECT 2 UNION SELECT 3");
		okp("SELECT 1 UNION ALL SELECT 2 UNION SELECT 3");
		okp("SELECT 1 MINUS SELECT 2 UNION SELECT 3");
		okp("SELECT 1 EXCEPT SELECT 2 UNION SELECT 3");
		okp("SELECT 1 INTERSECT SELECT 2 UNION SELECT 3");
		okp("SELECT 1 DIVIDE SELECT 2 UNION SELECT 3");
	}

	public void testParser004() {
		okp("(SELECT 1)");
		okp("(SELECT 1 UNION SELECT 2) INTERSECT SELECT 3");
	}

	public void test001() {
		eqp("SELECT 1", "(select ((1 . #f)) #f #t #f #t #f)");
	}

	public void test002() {
		eqp("SELECT 1 UNION SELECT 2",
				"(union (select ((1 . #f)) #f #t #f #t #f) (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 1 UNION ALL SELECT 2",
				"(union-all (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 1 MINUS SELECT 2",
				"(minus (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 1 EXCEPT SELECT 2",
				"(minus (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 1 INTERSECT SELECT 2",
				"(intersect (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 1 DIVIDE SELECT 2",
				"(divide (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f))");
	}

	public void testParser011() {
		eqp("SELECT 0 UNION SELECT 1 UNION SELECT 2",
				"(union (union (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION SELECT 1 UNION ALL SELECT 2",
				"(union-all (union (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION SELECT 1 MINUS SELECT 2",
				"(minus (union (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION SELECT 1 EXCEPT SELECT 2",
				"(minus (union (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION SELECT 1 INTERSECT SELECT 2",
				"(union (select ((0 . #f)) #f #t #f #t #f)" +
				" (intersect (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 UNION SELECT 1 DIVIDE SELECT 2",
				"(union (select ((0 . #f)) #f #t #f #t #f)" +
				" (divide (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");

		eqp("SELECT 0 UNION SELECT 1 UNION SELECT 2",
				"(union (union (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION ALL SELECT 1 UNION SELECT 2",
				"(union (union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 MINUS SELECT 1 UNION SELECT 2",
				"(union (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 EXCEPT SELECT 1 UNION SELECT 2",
				"(union (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 INTERSECT SELECT 1 UNION SELECT 2",
				"(union (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 UNION SELECT 2",
				"(union (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
	}

	public void testParser012() {
		eqp("SELECT 0 UNION ALL SELECT 1 UNION SELECT 2",
				"(union (union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION ALL SELECT 1 UNION ALL SELECT 2",
				"(union-all (union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION ALL SELECT 1 MINUS SELECT 2",
				"(minus (union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION ALL SELECT 1 EXCEPT SELECT 2",
				"(minus (union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION ALL SELECT 1 INTERSECT SELECT 2",
				"(union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (intersect (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 UNION ALL SELECT 1 DIVIDE SELECT 2",
				"(union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (divide (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");

		eqp("SELECT 0 UNION SELECT 1 UNION ALL SELECT 2",
				"(union-all (union (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION ALL SELECT 1 UNION ALL SELECT 2",
				"(union-all (union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 MINUS SELECT 1 UNION ALL SELECT 2",
				"(union-all (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 EXCEPT SELECT 1 UNION ALL SELECT 2",
				"(union-all (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 INTERSECT SELECT 1 UNION ALL SELECT 2",
				"(union-all (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 UNION ALL SELECT 2",
				"(union-all (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
	}

	public void testParser013() {
		eqp("SELECT 0 MINUS SELECT 1 UNION SELECT 2",
				"(union (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 MINUS SELECT 1 UNION ALL SELECT 2",
				"(union-all (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 MINUS SELECT 1 MINUS SELECT 2",
				"(minus (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 MINUS SELECT 1 EXCEPT SELECT 2",
				"(minus (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 MINUS SELECT 1 INTERSECT SELECT 2",
				"(minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (intersect (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 MINUS SELECT 1 DIVIDE SELECT 2",
				"(minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (divide (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");

		eqp("SELECT 0 UNION SELECT 1 MINUS SELECT 2",
				"(minus (union (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION ALL SELECT 1 MINUS SELECT 2",
				"(minus (union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 MINUS SELECT 1 MINUS SELECT 2",
				"(minus (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 EXCEPT SELECT 1 MINUS SELECT 2",
				"(minus (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 INTERSECT SELECT 1 MINUS SELECT 2",
				"(minus (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 MINUS SELECT 2",
				"(minus (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
	}

	public void testParser014() {
		eqp("SELECT 0 EXCEPT SELECT 1 UNION SELECT 2",
				"(union (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 EXCEPT SELECT 1 UNION ALL SELECT 2",
				"(union-all (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 EXCEPT SELECT 1 MINUS SELECT 2",
				"(minus (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 EXCEPT SELECT 1 EXCEPT SELECT 2",
				"(minus (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 EXCEPT SELECT 1 INTERSECT SELECT 2",
				"(minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (intersect (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 EXCEPT SELECT 1 DIVIDE SELECT 2",
				"(minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (divide (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");

		eqp("SELECT 0 UNION SELECT 1 EXCEPT SELECT 2",
				"(minus (union (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 UNION ALL SELECT 1 EXCEPT SELECT 2",
				"(minus (union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 MINUS SELECT 1 EXCEPT SELECT 2",
				"(minus (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 EXCEPT SELECT 1 EXCEPT SELECT 2",
				"(minus (minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 INTERSECT SELECT 1 EXCEPT SELECT 2",
				"(minus (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 EXCEPT SELECT 2",
				"(minus (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
	}

	public void testParser015() {
		eqp("SELECT 0 INTERSECT SELECT 1 UNION SELECT 2",
				"(union (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 INTERSECT SELECT 1 UNION ALL SELECT 2",
				"(union-all (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 INTERSECT SELECT 1 MINUS SELECT 2",
				"(minus (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 INTERSECT SELECT 1 EXCEPT SELECT 2",
				"(minus (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 INTERSECT SELECT 1 INTERSECT SELECT 2",
				"(intersect (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 INTERSECT SELECT 1 DIVIDE SELECT 2",
				"(divide (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");

		eqp("SELECT 0 UNION SELECT 1 INTERSECT SELECT 2",
				"(union (select ((0 . #f)) #f #t #f #t #f)" +
				" (intersect (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 UNION ALL SELECT 1 INTERSECT SELECT 2",
				"(union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (intersect (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 MINUS SELECT 1 INTERSECT SELECT 2",
				"(minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (intersect (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 EXCEPT SELECT 1 INTERSECT SELECT 2",
				"(minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (intersect (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 INTERSECT SELECT 1 INTERSECT SELECT 2",
				"(intersect (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 INTERSECT SELECT 2",
				"(intersect (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
	}

	public void testParser016() {
		eqp("SELECT 0 DIVIDE SELECT 1 UNION SELECT 2",
				"(union (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 UNION ALL SELECT 2",
				"(union-all (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 MINUS SELECT 2",
				"(minus (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 EXCEPT SELECT 2",
				"(minus (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 INTERSECT SELECT 2",
				"(intersect (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 DIVIDE SELECT 2",
				"(divide (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");

		eqp("SELECT 0 UNION SELECT 1 DIVIDE SELECT 2",
				"(union (select ((0 . #f)) #f #t #f #t #f)" +
				" (divide (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 UNION ALL SELECT 1 DIVIDE SELECT 2",
				"(union-all (select ((0 . #f)) #f #t #f #t #f)" +
				" (divide (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 MINUS SELECT 1 DIVIDE SELECT 2",
				"(minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (divide (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 EXCEPT SELECT 1 DIVIDE SELECT 2",
				"(minus (select ((0 . #f)) #f #t #f #t #f)" +
				" (divide (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
		eqp("SELECT 0 INTERSECT SELECT 1 DIVIDE SELECT 2",
				"(divide (intersect (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
		eqp("SELECT 0 DIVIDE SELECT 1 DIVIDE SELECT 2",
				"(divide (divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (select ((1 . #f)) #f #t #f #t #f))" +
				" (select ((2 . #f)) #f #t #f #t #f))");
	}

	public void testParser021() {
		eqp("SELECT 0 DIVIDE (SELECT 1 UNION SELECT 2)",
				"(divide (select ((0 . #f)) #f #t #f #t #f)" +
				" (union (select ((1 . #f)) #f #t #f #t #f)" +
				" (select ((2 . #f)) #f #t #f #t #f)))");
	}

}
