/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.relations;

import java.math.BigDecimal;
import java.math.MathContext;
import java.sql.SQLException;
import java.util.EnumMap;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.sqlcs.dml.SqlUnaryOperator;

public abstract class RelationUnaryOperator {

	static final MathContext MC = new MathContext(6);

	public static final RelationUnaryOperator
	MINUS = new RelationUnaryOperator() {

		@Override
		public Object op(Object a) throws SQLException {
			if(a instanceof BigDecimal) {
				return ((BigDecimal)a).negate();
			} else if(a.equals("")) {
				return a;
			} else {
				throw ErrorBundle.getDefault(10040, "unary -");
			}
		}

	};

	public static final RelationUnaryOperator
	PLUS = new RelationUnaryOperator() {

		@Override
		public Object op(Object a) throws SQLException {
			if(a instanceof BigDecimal) {
				return a;
			} else if(a.equals("")) {
				return a;
			} else {
				throw ErrorBundle.getDefault(10040, "unary +");
			}
		}

	};

	public static final RelationUnaryOperator
	NOT = new RelationUnaryOperator() {

		@Override
		public Object op(Object a) throws SQLException {
			return RelationExpression.value(
					RelationExpression.bool(a).not());
		}

	};

	public static final RelationUnaryOperator
	IS_NULL = new RelationUnaryOperator() {

		@Override
		public Object op(Object a) throws SQLException {
			return a.equals("");
		}

	};

	public static final RelationUnaryOperator
	IS_NOT_NULL = new RelationUnaryOperator() {

		@Override
		public Object op(Object a) throws SQLException {
			return !a.equals("");
		}

	};

	private static
	EnumMap<SqlUnaryOperator, RelationUnaryOperator> map;

	static {
		map = new EnumMap<SqlUnaryOperator, RelationUnaryOperator>(
				SqlUnaryOperator.class);
		map.put(SqlUnaryOperator.MINUS, MINUS);
		map.put(SqlUnaryOperator.PLUS, PLUS);
		map.put(SqlUnaryOperator.NOT, NOT);
		map.put(SqlUnaryOperator.IS_NULL, IS_NULL);
		map.put(SqlUnaryOperator.IS_NOT_NULL, IS_NOT_NULL);
	}

	/**
	 * 
	 * @param s
	 * @return
	 */
	public static RelationUnaryOperator get(SqlUnaryOperator s) {
		return map.get(s);
	}

	/**
	 * 
	 * @param a
	 * @return
	 */
	public abstract Object op(Object a) throws SQLException;

}
