/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.relations;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.morilib.db.misc.ErrorBundle;

public class IsolatedRelation extends AbstractRelation {

	private Relation rel;
	private String name;

	public IsolatedRelation(Relation r, String n) {
		rel = r;
		name = n;
	}

	public String getName() {
		return name;
	}

	@Override
	public List<String> getColumnNames() {
		List<String> l = new ArrayList<String>(rel.getColumnNames());

		for(String s : rel.getColumnNames()) {
			l.add(name + "." + s);
		}
		return l;
	}

	@Override
	public RelationCursor iterator() {
		final RelationCursor c = rel.iterator();

		return new RelationCursor() {

			@Override
			public boolean hasNext() {
				return c.hasNext();
			}

			@Override
			public RelationTuple next() throws IOException, SQLException {
				final RelationTuple t = c.next();

				return new RelationTuple() {

					@Override
					public Object get(String name) throws SQLException {
						int i = name.indexOf('.');

						if(i < 0) {
							return t.get(name);
						} else if(name.substring(0, i).equals(name)) {
							return t.get(name.substring(i + 1));
						} else {
							throw ErrorBundle.getDefault(10009, name);
						}
					}

					@Override
					public RelationTuple copy() throws SQLException {
						return new DefaultRelationTuple(toMap());
					}

					@Override
					public Map<String, Object> toMap() throws SQLException {
						Map<String, Object> m, s;

						m = new LinkedHashMap<String, Object>();
						m.putAll(s = t.toMap());
						for(String v : s.keySet()) {
							m.put(name + "." + v, s.get(v));
						}
						return m;
					}

				};
			}

		};
	}

}
