------------------------------------------------------------------------------
-- Copyright 2018 Levashev Ivan Aleksandrovich                              --
--                                                                          --
-- Licensed under the Apache License, Version 2.0 (the "License");          --
-- you may not use this file except in compliance with the License.         --
-- You may obtain a copy of the License at                                  --
--                                                                          --
--     http://www.apache.org/licenses/LICENSE-2.0                           --
--                                                                          --
-- Unless required by applicable law or agreed to in writing, software      --
-- distributed under the License is distributed on an "AS IS" BASIS,        --
-- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. --
-- See the License for the specific language governing permissions and      --
-- limitations under the License.                                           --
------------------------------------------------------------------------------

with Ada.Text_IO;

with System.Address_Image;

with Referencing.Debug;

package body Referencing.Tester is

   -----------
   -- Image --
   -----------

   function Image
     (Item : Referencing.Types.Operations.Referenced_Access)
     return String is
     (System.Address_Image
        (Referencing.Types.Operations.To_Address
           (Item)));

   -------------------------------
   -- Sample_Reference.Assigned --
   -------------------------------

   function Assigned
     (Object : Sample_Reference'Class)
     return Boolean is
     (Operations.Assigned (Object));

   -----------------------------
   -- Sample_Reference.Assign --
   -----------------------------

   procedure Assign
     (Target : in out Sample_Reference'Class;
      Source : Sample_Reference'Class) is
   begin
      Operations.Assign (Target, Source);
   end Assign;

   ---------------------------
   -- Sample_Reference.Move --
   ---------------------------

   procedure Move
     (Target : in out Sample_Reference'Class;
      Source : in out Sample_Reference'Class) is
   begin
      Operations.Move (Target, Source);
   end Move;

   ---------------------------------------
   -- Sample_Reference.Construct_Result --
   ---------------------------------------

   function Construct_Result
     return Sample_Limited_Reference is
     (Create_Limited_Reference);

   -----------------------------
   -- Sample_Reference.Create --
   -----------------------------

   function Create
     (Value : Integer)
     return Sample_Limited_Reference is
   begin
      return Result : Sample_Limited_Reference := Create_Limited_Reference do
         declare
            New_Object : not null Types.Sample_Access := new Types.Sample_Referenced;
         begin
            Operations.Set (Result, New_Object);
            Types.Create (New_Object, Value);
         end;
      end return;
   end Create;

   ---------------------------
   -- Sample_Reference.Plus --
   ---------------------------

   function Plus (Left, Right : Sample_Reference'Class) return Sample_Limited_Reference is
   begin
      return Types.Plus (Operations.Get (Left), Right);
   end Plus;

   -----------
   -- Types --
   -----------

   package body Types is

      ----------------------------
      -- Sample_Referenced.Self --
      ----------------------------

      function Self
        (Object : access Sample_Referenced)
        return Sample_Limited_Reference is
        (Tester.Create_And_Retain (Object.all'Unchecked_Access))
        with Inline_Always;

      ------------------------------
      -- Sample_Referenced.Create --
      ------------------------------

      procedure Create
        (Object : access Sample_Referenced;
         Value : Integer) is
      begin
         Object.Value_Field := Value;
         begin
            Ada.Text_IO.Put_Line
              ("Creating " & Image (Object.all'Unchecked_Access) &
               " with value" & Integer'Image (Value));
         exception when others => null; end;
      end Create;

      ----------------------------
      -- Sample_Referenced.Plus --
      ----------------------------

      function Plus
        (Left : access Sample_Referenced;
         Right : Sample_Reference'Class)
        return Sample_Limited_Reference
      is
         Right_Access : constant Sample_Access := Operations.Get (Right);
      begin
         return Create (Left.Value_Field + Right_Access.Value_Field);
      end Plus;

      ---------
      -- Get --
      ---------

      function Get
        (Object : in Sample_Reference'Class)
        return Sample_Access is
        (Operations.Get (Object));

      -----------------------
      -- Create_And_Retain --
      -----------------------

      function Create_And_Retain
        (Object : Sample_Access)
        return Sample_Limited_Reference is
        (Tester.Create_And_Retain (Object));

      ----------------------------------
      -- Sample_Referenced.Initialize --
      ----------------------------------

      overriding
      procedure Initialize (Object : in out Sample_Referenced) is
      begin
         declare
            use Sample_Referenced_Parents;
         begin
            Initialize (Sample_Referenced_Parent (Object));
         end;

         begin
            Ada.Text_IO.Put_Line
              ("Initializing " & Image (Object'Unchecked_Access));
         exception when others => null; end;
      end Initialize;

      --------------------------------
      -- Sample_Referenced.Finalize --
      --------------------------------

      overriding
      procedure Finalize (Object : in out Sample_Referenced) is
      begin
         begin
            Ada.Text_IO.Put_Line
              ("Finalizing " & Image (Object'Unchecked_Access) &
               " with value" & Integer'Image (Object.Value_Field));
         exception when others => null; end;

         declare
            use Sample_Referenced_Parents;
         begin
            Finalize (Sample_Referenced_Parent (Object));
         end;
      end Finalize;

   end Types;

   ------------
   -- Upcast --
   ------------

   function Upcast
     (Object : Types.Sample_Access)
     return Referencing.Types.Operations.Referenced_Access is
     (Referencing.Types.Operations.Referenced_Access (Object));

begin
   Referencing.Debug.Put_Line := Ada.Text_IO.Put_Line'Access;
end Referencing.Tester;
