------------------------------------------------------------------------------
-- Copyright 2018 Levashev Ivan Aleksandrovich                              --
--                                                                          --
-- Licensed under the Apache License, Version 2.0 (the "License");          --
-- you may not use this file except in compliance with the License.         --
-- You may obtain a copy of the License at                                  --
--                                                                          --
--     http://www.apache.org/licenses/LICENSE-2.0                           --
--                                                                          --
-- Unless required by applicable law or agreed to in writing, software      --
-- distributed under the License is distributed on an "AS IS" BASIS,        --
-- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. --
-- See the License for the specific language governing permissions and      --
-- limitations under the License.                                           --
------------------------------------------------------------------------------

with System;
with System.Address_To_Access_Conversions;

with Referencing.Types.Operations;

package body Referencing.References is

   ----------------
   -- Operations --
   ----------------

   package body Operations is

      --------------
      -- Assigned --
      --------------

      function Assigned
        (Object : Reference'Class)
        return Boolean is
        (Reference_Base (Object).Internal_Access /= Null_Access);

      ------------
      -- Assign --
      ------------

      procedure Assign
        (Target : in out Reference'Class;
         Source : Reference'Class)
      is
         Saved_Source_Access : constant Classwide_Access :=
           Reference_Base (Source).Internal_Access;
         Saved_Target_Access : constant Classwide_Access :=
           Reference_Base (Target).Internal_Access;
      begin
         if Saved_Target_Access = Saved_Source_Access then
            return;
         end if;

         Reference_Base (Target).Internal_Access :=
           Saved_Source_Access;

         if Saved_Source_Access /= Null_Access then
            Types.Operations.Retain (Upcast (Saved_Source_Access).all);
         end if;

         if Saved_Target_Access /= Null_Access then
            declare
               Upcasted : Types.Operations.Referenced_Access := Upcast (Saved_Target_Access);
            begin
               if Types.Operations.Release (Upcasted.all) then
                  Referencing.Types.Operations.Free (Upcasted);
               end if;
            end;
         end if;
      end Assign;

      ----------
      -- Move --
      ----------

      procedure Move
        (Target : in out Reference'Class;
         Source : in out Reference'Class)
      is
         Saved_Target_Access : constant Classwide_Access :=
           Reference_Base (Target).Internal_Access;
      begin
         Reference_Base (Target).Internal_Access :=
           Reference_Base (Source).Internal_Access;
         Reference_Base (Source).Internal_Access :=
           Null_Access;

         if Saved_Target_Access /= Null_Access then
            declare
               Upcasted : Types.Operations.Referenced_Access := Upcast (Saved_Target_Access);
            begin
               if Types.Operations.Release (Upcasted.all) then
                  Referencing.Types.Operations.Free (Upcasted);
               end if;
            end;
         end if;
      end Move;

      ---------
      -- Get --
      ---------

      function Get
        (Object : in Reference'Class)
        return Classwide_Access is
        (Reference_Base (Object).Internal_Access);

      ---------
      -- Set --
      ---------

      procedure Set
        (Object : in out Reference'Class;
         Item : Classwide_Access) is
      begin
         Reference_Base (Object).Internal_Access := Item;
      end Set;

      --------------------
      -- Set_And_Retain --
      --------------------

      procedure Set_And_Retain
        (Object : in out Reference'Class;
         Item : Classwide_Access)
      is
         Saved_Object_Access : constant Classwide_Access := Get (Object);
      begin
         if Item = Saved_Object_Access then
            return;
         end if;
         
         if Item /= Null_Access then
            Types.Operations.Retain (Upcast (Item).all);
         end if;

         Set (Object, Item);

         if Saved_Object_Access /= Null_Access then
            declare
               Upcasted : Types.Operations.Referenced_Access := Upcast (Saved_Object_Access);
            begin
               if Types.Operations.Release (Upcasted.all) then
                  Referencing.Types.Operations.Free (Upcasted);
               end if;
            end;
         end if;
      end Set_And_Retain;

      type Mutable_Limited_Reference is limited record
         Data : access Reference;
         Wrapped : aliased Reference;
      end record;
      
      pragma Assert (Mutable_Limited_Reference'Size = Limited_Reference_Base'Size);
      pragma Assert (Mutable_Limited_Reference'Alignment = Limited_Reference_Base'Alignment);

      package Limited_Reference_Conversions is new
        System.Address_To_Access_Conversions
        (Limited_Reference_Base);
      subtype Limited_Reference_Access is Limited_Reference_Conversions.Object_Pointer;

      ------------------------------
      -- Create_Limited_Reference --
      ------------------------------
      
      function Create_Limited_Reference
        return Limited_Reference_Base
      is
         Dummy : aliased Reference with Import, Address => Dummy_Structure'Address;
      begin
         return Result : aliased Limited_Reference_Base (Data => Dummy'Unchecked_Access) do
            declare
               Mutable_Result : Mutable_Limited_Reference
                 with Import,
                      Address => Limited_Reference_Conversions.To_Address
                                   (Result'Unchecked_Access);
            begin
               pragma Assert (Mutable_Result.Data = Dummy'Unchecked_Access);
               Mutable_Result.Data := Result.Wrapped'Unchecked_Access;
            end;
         end return;
      end Create_Limited_Reference;

      -----------------------
      -- Create_And_Retain --
      -----------------------

      function Create_And_Retain (Object : Classwide_Access) return Limited_Reference_Base is
      begin
         return Result : Limited_Reference_Base := Create_Limited_Reference do
            if Object /= Null_Access then
               Types.Operations.Retain (Upcast (Object).all);
            end if;

            Operations.Set (Result, Object);
         end return;
      end Create_And_Retain;

   end Operations;

   ----------------------
   -- Reference.Adjust --
   ----------------------

   overriding
   procedure Adjust (Object : in out Reference_Base) is
      Internal_Access : Classwide_Access renames Object.Internal_Access;
   begin
      if Internal_Access /= Null_Access then
         Types.Operations.Retain (Upcast (Internal_Access).all);
      end if;
   end Adjust;

   ------------------------
   -- Reference.Finalize --
   ------------------------

   overriding
   procedure Finalize (Object : in out Reference_Base) is
      Internal_Access : Classwide_Access renames Object.Internal_Access;
   begin
      if Internal_Access /= Null_Access then
         declare
            Upcasted : Types.Operations.Referenced_Access := Upcast (Internal_Access);
         begin
            Internal_Access := Null_Access;

            if Types.Operations.Release (Upcasted.all) then
               Referencing.Types.Operations.Free (Upcasted);
            end if;
         end;
      end if;
   end Finalize;

end Referencing.References;
