------------------------------------------------------------------------------
-- Copyright 2019 Levashev Ivan Aleksandrovich                              --
--                                                                          --
-- Licensed under the Apache License, Version 2.0 (the "License");          --
-- you may not use this file except in compliance with the License.         --
-- You may obtain a copy of the License at                                  --
--                                                                          --
--     http://www.apache.org/licenses/LICENSE-2.0                           --
--                                                                          --
-- Unless required by applicable law or agreed to in writing, software      --
-- distributed under the License is distributed on an "AS IS" BASIS,        --
-- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. --
-- See the License for the specific language governing permissions and      --
-- limitations under the License.                                           --
------------------------------------------------------------------------------

--  This package provides atomic counter for AdaMagic combined with
--  C/C++ compiler supporting Itanium ABI atomics. Support can be further 
--  enhanced by tweaking "Ada_Magic_Forward.Atomic_Counters.h".
--
--  Not intended to be with'ed directly. Instead, System.Atomic_Counters
--  facade should be used just like in GNAT. Read it for more comments about 
--  the design.

package Ada_Magic_Forward.Atomic_Counters is

   pragma Pure;
   pragma Preelaborate;

   type Atomic_Counter is limited private;
   --  Type for atomic counter objects. Note, initial value of the counter is
   --  one. This allows using an atomic counter as member of record types when
   --  object of these types are created at library level in preelaborable
   --  compilation units.
   --
   --  Atomic_Counter is declared as private limited type to provide highest
   --  level of protection from unexpected use. All available operations are
   --  declared below, and this set should be as small as possible.
   --  Increment/Decrement operations for this type raise Program_Error on
   --  platforms not supporting the atomic primitives.

   procedure Increment (Item : in out Atomic_Counter);
   --  Increments value of atomic counter.

   function Decrement (Item : in Atomic_Counter) return Boolean;
   --  Decrements value of atomic counter, returns True when value reach zero
   --  "in" for source-level compatibility with GNAT
   --  In reality works as "in out"

   function Is_One (Item : Atomic_Counter) return Boolean;
   --  Returns True when value of the atomic counter is one

   procedure Initialize (Item : out Atomic_Counter);
   --  Initialize counter by setting its value to one. This subprogram is
   --  intended to be used in special cases when the counter object cannot be
   --  initialized in standard way.

   package Unsigned is

      type Atomic_Unsigned is mod 2 ** 32;
      pragma Atomic (Atomic_Unsigned);
      --  Modular compatible atomic unsigned type.
      --  Increment/Decrement operations for this type are atomic only on
      --  supported platforms. See top of the file.

      procedure Increment
        (Item : in out Atomic_Unsigned);
      --  Increments value of atomic counter

      function Decrement
        (Item : in Atomic_Unsigned) return Boolean;
      --  Same trick with "in" as "in out" to try to be source-level
      --  compatible with GNAT

      --  The "+" and "-" abstract routine provided below to disable BT := BT + 1
      --  constructions.

      function "+"
        (Left, Right : Atomic_Unsigned) return Atomic_Unsigned is abstract;

      function "-"
        (Left, Right : Atomic_Unsigned) return Atomic_Unsigned is abstract;

   private

      type Include_Atomic_Counters is new Integer;
      pragma Import (C, Include_Atomic_Counters, "#include <Source/Ada_Magic_Forward.Atomic_Counters.h>", "int");

      pragma Import (Intrinsic, Increment, "Ada_magic_forward_Atomic_counters_Unsigned_Increment");
      pragma Import (Intrinsic, Decrement, "Ada_magic_forward_Atomic_counters_Unsigned_Decrement");

   end Unsigned;

   package Unsigned_Addon is

      type Atomic_Unsigned is new Unsigned.Atomic_Unsigned;

      procedure Decrement
        (Item : in out Atomic_Unsigned);
      --  Decrements value of atomic counter

   private
      pragma Import (Intrinsic, Decrement, "Ada_magic_forward_Atomic_counters_Unsigned_addon_Decrement");
   end Unsigned_Addon;

private

   type Atomic_Counter is record
      Value : aliased Unsigned_Addon.Atomic_Unsigned := 1;
      pragma Atomic (Value);
   end record;

   pragma Import (Intrinsic, Increment,  "Ada_magic_forward_Atomic_counters_Increment");
   pragma Import (Intrinsic, Decrement,  "Ada_magic_forward_Atomic_counters_Decrement");
   pragma Import (Intrinsic, Is_One,     "Ada_magic_forward_Atomic_counters_Is_one");
   pragma Import (Intrinsic, Initialize, "Ada_magic_forward_Atomic_counters_Initialize");

end Ada_Magic_Forward.Atomic_Counters;
