------------------------------------------------------------------------------
-- Copyright 2019 Levashev Ivan Aleksandrovich                              --
--                                                                          --
-- Licensed under the Apache License, Version 2.0 (the "License");          --
-- you may not use this file except in compliance with the License.         --
-- You may obtain a copy of the License at                                  --
--                                                                          --
--     http://www.apache.org/licenses/LICENSE-2.0                           --
--                                                                          --
-- Unless required by applicable law or agreed to in writing, software      --
-- distributed under the License is distributed on an "AS IS" BASIS,        --
-- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. --
-- See the License for the specific language governing permissions and      --
-- limitations under the License.                                           --
------------------------------------------------------------------------------

with Ada_Magic_Forward.Atomic_Counters;

--  This package provides atomic counter for AdaMagic combined with
--  C/C++ compiler supporting Itanium ABI atomics. Support can be further 
--  enhanced by tweaking "Ada_Magic_Forward.Atomic_Counters.h".
--
--  This package tries to be source-level compatible with GNAT's 
--  System.Atomic_Counters, but there are notable differences:
--
--  1. Functions with "in out" parameters are paradigm shift in Ada 2012.
--  GNAT version relies on it. AdaMagic stuck little to no further from 
--  Ada 95. Workarounds: parameters are semantically read-only ("in") in
--  AdaMagic. It is possible to decrement read-only counter in AdaMagic 
--  version.
--
--  2. Atomic_Unsigned is not initialized by 0. Default_Value is a
--  comparably recent pragma. AdaMagic does not support it.
--
--  3. Increment and Decrement for Atomic_Unsigned do not require aliased 
--  parameter because Ada 95 has no this feature.
--
--  In all cases AdaMagic may compile some code that GNAT would reject. This 
--  matches the intended usage: the same code is compiled by GNAT and 
--  AdaMagic, and source-level compatibility matters more than non-existent
--  encapsulation issues already rejected by GNAT.
--
--  Real bindings were moved to Ada_Magic_Forward.Atomic_Counters due to lack 
--  of aspects in Ada 95. Compared to aspect Import, pragma Import cannot tell 
--  between overloaded names, thus they have to be in different packages.
--  Ada_Magic_Forward.Atomic_Counters does a job of adding primitive imported 
--  operations in different subpackages, and System.Atomic_Counters presents a
--  unified facade for source-level compatibility.

package System.Atomic_Counters is

   pragma Pure;
   pragma Preelaborate;

   type Atomic_Counter is
     new Ada_Magic_Forward.Atomic_Counters.Atomic_Counter;
   --  Type for atomic counter objects. Note, initial value of the counter is
   --  one. This allows using an atomic counter as member of record types when
   --  object of these types are created at library level in preelaborable
   --  compilation units.
   --
   --  Atomic_Counter is declared as private limited type to provide highest
   --  level of protection from unexpected use. All available operations are
   --  declared below, and this set should be as small as possible.
   --  Increment/Decrement operations for this type raise Program_Error on
   --  platforms not supporting the atomic primitives.
   --
   --  "is new" copies the following operations from
   --  Ada_Magic_Forward.Atomic_Counters.

   --  procedure Increment (Item : in out Atomic_Counter);
   --  --  Increments value of atomic counter.

   --  function Decrement (Item : in Atomic_Counter) return Boolean;
   --  --  Decrements value of atomic counter, returns True when value reach zero
   --  --  "in" for source-level compatibility with GNAT
   --  --  In reality works as "in out"

   --  function Is_One (Item : Atomic_Counter) return Boolean;
   --  --  Returns True when value of the atomic counter is one

   --  procedure Initialize (Item : out Atomic_Counter);
   --  --  Initialize counter by setting its value to one. This subprogram is
   --  --  intended to be used in special cases when the counter object cannot be
   --  --  initialized in standard way.

   type Atomic_Unsigned is
     new Ada_Magic_Forward.Atomic_Counters.Unsigned_Addon.Atomic_Unsigned;
   --  Modular compatible atomic unsigned type.
   --  Increment/Decrement operations for this type are atomic only on
   --  supported platforms. See top of the file.
   --
   --  In contrast to GNAT's Atomic_Unsigned, there is no pragma Default_Value!
   --  AdaMagic does not support it.
   --
   --  "is new" copies the following operations from
   --  Ada_Magic_Forward.Atomic_Counters.

   --  procedure Increment
   --    (Item : in out Atomic_Unsigned);
   --  --  Increments value of atomic counter

   --  function Decrement
   --    (Item : in Atomic_Unsigned) return Boolean;
   --  --  Same trick with "in" as "in out" to try to be source-level
   --  --  compatible with GNAT

   --  procedure Decrement
   --    (Item : in out Atomic_Unsigned);
   --  --  Decrements value of atomic counter

   --  --  The "+" and "-" abstract routine provided below to disable BT := BT + 1
   --  --  constructions.

   --  function "+"
   --    (Left, Right : Atomic_Unsigned) return Atomic_Unsigned is abstract;

   --  function "-"
   --    (Left, Right : Atomic_Unsigned) return Atomic_Unsigned is abstract;

end System.Atomic_Counters;
