/*****************************************************************************
 *    Licensed to the Apache Software Foundation (ASF) under one
 *    or more contributor license agreements.  See the NOTICE file
 *    distributed with this work for additional information
 *    regarding copyright ownership.  The ASF licenses this file
 *    to you under the Apache License, Version 2.0 (the
 *    "License"); you may not use this file except in compliance
 *    with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing,
 *    software distributed under the License is distributed on an
 *    "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *    KIND, either express or implied.  See the License for the
 *    specific language governing permissions and limitations
 *    under the License.
 *
 * UART
 * $Id: rbtUART.c 29 2012-11-17 18:28:25Z xeerda $
 *****************************************************************************/
#include "rbtUART.h"

#define TXBUFFSIZE		(64)
#define RXBUFFSIZE		(8)
static char txBuff[TXBUFFSIZE];
static char rxBuff[RXBUFFSIZE];
static volatile short nTxRD = 0;
static volatile short nTxWR = 0;
static volatile short nRxRD = 0;
static volatile short nRxWR = 0;

static RBTOS_LockT lockRX = RBTOS_INITIALIZED_LOCK_VALUE;
static RBTOS_LockT lockTX = RBTOS_INITIALIZED_LOCK_VALUE;
static RBTOS_EventT evtRX = RBTOS_INITIALIZED_EVENT_VALUE;
static RBTOS_EventT evtTX = RBTOS_INITIALIZED_EVENT_VALUE;

static RBTOS_DeviceT sDevUART;
RBTOS_DeviceT *devUART;

static int uart_chkready(void)
{
	return 1;
}

/**
 * UART ǂݏo.
 * <p>
 * length  1 ȏw肵ꍇCȂƂ 1 擾܂ŃubN܂.
 * @param xbuff ǂݏoi[obt@.
 * @param length ǂݏo.
 * @return ۂɓǂݏo.
 */
static int uart_read(char *xbuff, int length)
{
	int rdsz = 0;
	RBTOS_Lock_enter(&lockRX);
	while (length > 0) {
		short rxAvailSz = nRxWR - nRxRD;
		if (rxAvailSz < 0) {
			rxAvailSz += RXBUFFSIZE;
		}
		if (rxAvailSz != 0) {
			if (rxAvailSz > length) {
				rxAvailSz = length;
			}
			if (rxAvailSz > (RXBUFFSIZE - nRxRD)) {
				rxAvailSz = (RXBUFFSIZE - nRxRD);
			}
			memcpy(xbuff, (void*) (rxBuff + nRxRD), rxAvailSz);
			xbuff += rxAvailSz;
			nRxRD = (nRxRD + rxAvailSz) & (RXBUFFSIZE - 1);
			length -= rxAvailSz;
			rdsz += rxAvailSz;
		} else {
			if (rdsz != 0) {
				length = 0;
			} else {
				RBTOS_Event_wait(&evtRX, 0);
			}
		}
	}
	RBTOS_Lock_leave(&lockRX);
	return rdsz;
}


/**
 * UART ւ̏.
 * <p>
 * length ׂđMobt@ɏނ܂ŃubN܂.
 * @param xbuff ޕw肵܂.
 * @param length ޕw肵܂.
 * @return ۂɏ񂾕 (= length)
 */
static int uart_write(const char *xbuff, int length)
{
	if (fCritical) {
		while (nTxWR != nTxRD) {
			while (!UARTTransmitterIsReady(UART2)) {
			}
			UARTSendDataByte(UART2, txBuff[nTxRD]);
			nTxRD = (nTxRD + 1) & (TXBUFFSIZE - 1);
		}
		int wrsz = length;
		for (; length != 0; length--) {
			while (U2STAbits.UTXBF != 0) {
			}
			// Mobt@Ƀf[^ȂCTX FIFO ɏ
			U2TXREG = *xbuff++;
		}
		return wrsz;
	}
	int wrsz = 0;
	RBTOS_Lock_enter(&lockTX);
	while (length > 0) {
		
		// Mobt@̋󂫃oCgZo.
		short txFreeSz = nTxRD - nTxWR - 1;
		if (txFreeSz < 0) {
			txFreeSz += TXBUFFSIZE;
		}
		if (txFreeSz != 0) {
			// Mobt@ɋ󂫂ȂC.
			short onesz = TXBUFFSIZE - nTxWR;
			if (onesz > txFreeSz) {
				onesz = txFreeSz;
			}
			if (onesz > length) {
				onesz = length;
			}
			memcpy(txBuff + nTxWR, xbuff, onesz);
			xbuff += onesz;
			length -= onesz;
			wrsz += onesz;
			
			nTxWR = (nTxWR + onesz) & (TXBUFFSIZE - 1);
			INTEnable(INT_SOURCE_UART_TX(UART2), INT_ENABLED);
		} else {
			RBTOS_Event_wait(&evtTX, 0);
		}
	}
	RBTOS_Lock_leave(&lockTX);
	return wrsz;
}

// Mobt@̗Lf[^oCg̊mF
static int uart_avail(void)
{
	short rxAvailSz;
	rxAvailSz = nRxWR - nRxRD;
	if (rxAvailSz < 0) {
		rxAvailSz += RXBUFFSIZE;
	}
	return rxAvailSz;
}


// 
void uart_init(void)
{
	U2RXR = 0b0100; // RPB8
	RPB9R = 0b0010; // U2TX
	
    UARTConfigure(UART2, UART_ENABLE_PINS_TX_RX_ONLY);
    UARTSetFifoMode(UART2, UART_INTERRUPT_ON_TX_NOT_FULL | UART_INTERRUPT_ON_RX_NOT_EMPTY);
    UARTSetLineControl(UART2, UART_DATA_SIZE_8_BITS | UART_PARITY_NONE | UART_STOP_BITS_1);
    UARTSetDataRate(UART2, 40000000UL, 38400);
    UARTEnable(UART2, UART_ENABLE_FLAGS(UART_PERIPHERAL | UART_RX | UART_TX));
	
    INTSetVectorPriority(INT_VECTOR_UART(UART2), INT_PRIORITY_LEVEL_2);
    INTSetVectorSubPriority(INT_VECTOR_UART(UART2), INT_SUB_PRIORITY_LEVEL_0);
	INTEnable(INT_SOURCE_UART_RX(UART2), INT_ENABLED);
	INTEnable(INT_SOURCE_UART_TX(UART2), INT_DISABLED);
	
	sDevUART.d_read = uart_read;
	sDevUART.d_write = uart_write;
	sDevUART.d_avail = uart_avail;
	sDevUART.d_chkready = uart_chkready;
	
	devUART = &sDevUART;
	
	RBTOS_Event_create(&sDevUART.evtDevice);
}

void __ISR(_UART2_VECTOR, ipl2) IntUart2HandlerWrapper(void);


void IntUart2Handler(void)
//void __ISR(_UART2_VECTOR, ipl2) IntUart2Handler(void)
{
	if (INTGetFlag(INT_SOURCE_UART_RX(UART2))) {
		INTClearFlag(INT_SOURCE_UART_RX(UART2));
		
		if (U2STAbits.OERR != 0) {
			U2STAbits.OERR = 0;
		}
		// Mobt@̋󂫃oCg̎Zo.
		signed char iv = nRxRD - nRxWR - 1;
		if (iv < 0) {
			iv += RXBUFFSIZE;
		}
		for (; UARTReceivedDataIsAvailable(UART2);) {
			char ch = UARTGetDataByte(UART2);
			if (iv != 0) {
				// Mobt@ɋ󂫂ȂΓǂݎ̂.
				rxBuff[nRxWR] = ch;
				nRxWR = (nRxWR + 1) & (RXBUFFSIZE - 1);
				iv--;
			}
		}
		iRBTOS_Event_set(&evtRX);
	}
	if (INTGetFlag(INT_SOURCE_UART_TX(UART2))) {
		if (nTxWR != nTxRD) {
			INTClearFlag(INT_SOURCE_UART_TX(UART2));
			
			// Mobt@̏[UoCǧvZ.
			signed char iv = nTxWR - nTxRD;
			if (iv < 0) {
				iv += TXBUFFSIZE;
			}
			while ((iv != 0) && UARTTransmitterIsReady(UART2)) {
				// Mobt@Ƀf[^ȂCTX FIFO ɏ
				UARTSendDataByte(UART2, txBuff[nTxRD]);
				nTxRD = (nTxRD + 1) & (TXBUFFSIZE - 1);
				iv--;
			}
		} else {
			// Mobt@Ƀf[^ȂȂCM~
			INTEnable(INT_SOURCE_UART_TX(UART2), INT_DISABLED);
		}
		iRBTOS_Event_set(&evtTX);
	}
}
