/*****************************************************************************
 *    Licensed to the Apache Software Foundation (ASF) under one
 *    or more contributor license agreements.  See the NOTICE file
 *    distributed with this work for additional information
 *    regarding copyright ownership.  The ASF licenses this file
 *    to you under the Apache License, Version 2.0 (the
 *    "License"); you may not use this file except in compliance
 *    with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing,
 *    software distributed under the License is distributed on an
 *    "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *    KIND, either express or implied.  See the License for the
 *    specific language governing permissions and limitations
 *    under the License.
 *
 * UART
 * $Id: rbtUART.c 24 2012-11-13 16:56:50Z xeerda $
 *****************************************************************************/
#include "rbtUART.h"

#define TXBUFFSIZE		(64)
#define RXBUFFSIZE		(8)
static char txBuff[TXBUFFSIZE];
static char rxBuff[RXBUFFSIZE];
static volatile short nTxRD = 0;
static volatile short nTxWR = 0;
static volatile short nRxRD = 0;
static volatile short nRxWR = 0;

static RBTOS_LockT lockRX = RBTOS_INITIALIZED_LOCK_VALUE;
static RBTOS_LockT lockTX = RBTOS_INITIALIZED_LOCK_VALUE;
static RBTOS_EventT evtRX = RBTOS_INITIALIZED_EVENT_VALUE;
static RBTOS_EventT evtTX = RBTOS_INITIALIZED_EVENT_VALUE;

static RBTOS_DeviceT sDevUART;
RBTOS_DeviceT *devUART;

static int uart_chkready(void)
{
	return 1;
}

/**
 * UART ǂݏo.
 * <p>
 * length  1 ȏw肵ꍇCȂƂ 1 擾܂ŃubN܂.
 * @param xbuff ǂݏoi[obt@.
 * @param length ǂݏo.
 * @return ۂɓǂݏo.
 */
static int uart_read(char *xbuff, int length)
{
	int rdsz = 0;
	RBTOS_Lock_enter(&lockRX);
	while (length > 0) {
		short rxAvailSz = nRxWR - nRxRD;
		if (rxAvailSz < 0) {
			rxAvailSz += RXBUFFSIZE;
		}
		if (rxAvailSz != 0) {
			if (rxAvailSz > length) {
				rxAvailSz = length;
			}
			if (rxAvailSz > (RXBUFFSIZE - nRxRD)) {
				rxAvailSz = (RXBUFFSIZE - nRxRD);
			}
			memcpy(xbuff, (void*) (rxBuff + nRxRD), rxAvailSz);
			xbuff += rxAvailSz;
			nRxRD = (nRxRD + rxAvailSz) & (RXBUFFSIZE - 1);
			length -= rxAvailSz;
			rdsz += rxAvailSz;
		} else {
			if (rdsz != 0) {
				length = 0;
			} else {
				RBTOS_Event_wait(&evtRX, 0);
			}
		}
	}
	RBTOS_Lock_leave(&lockRX);
	return rdsz;
}


/**
 * UART ւ̏.
 * <p>
 * length ׂđMobt@ɏނ܂ŃubN܂.
 * @param xbuff ޕw肵܂.
 * @param length ޕw肵܂.
 * @return ۂɏ񂾕 (= length)
 */
static int uart_write(const char *xbuff, int length)
{
	if (fCritical) {
		while (nTxWR != nTxRD) {
			while (U1STAbits.TRMT == 0) {
			}
			U1TXREG = txBuff[nTxRD];
			nTxRD = (nTxRD + 1) & (TXBUFFSIZE - 1);
		}
		int wrsz = length;
		for (; length != 0; length--) {
			while (U1STAbits.TRMT == 0) {
			}
			// Mobt@Ƀf[^ȂCTX FIFO ɏ
			U1TXREG = *xbuff++;
		}
		return wrsz;
	}
	int wrsz = 0;
	RBTOS_Lock_enter(&lockTX);
	while (length > 0) {
		
		// Mobt@̋󂫃oCgZo.
		short txFreeSz = nTxRD - nTxWR - 1;
		if (txFreeSz < 0) {
			txFreeSz += TXBUFFSIZE;
		}
		if (txFreeSz != 0) {
			// Mobt@ɋ󂫂ȂC.
			short onesz = TXBUFFSIZE - nTxWR;
			if (onesz > txFreeSz) {
				onesz = txFreeSz;
			}
			if (onesz > length) {
				onesz = length;
			}
			memcpy(txBuff + nTxWR, xbuff, onesz);
			xbuff += onesz;
			length -= onesz;
			wrsz += onesz;
			
			nTxWR = (nTxWR + onesz) & (TXBUFFSIZE - 1);
			IEC0bits.U1TXIE = 1;
		} else {
			RBTOS_Event_wait(&evtTX, 0);
		}
	}
	RBTOS_Lock_leave(&lockTX);
	return wrsz;
}

// Mobt@̗Lf[^oCg̊mF
static int uart_avail(void)
{
	short rxAvailSz;
	rxAvailSz = nRxWR - nRxRD;
	if (rxAvailSz < 0) {
		rxAvailSz += RXBUFFSIZE;
	}
	return rxAvailSz;
}


// 
void uart_init(void)
{
	TRISBbits.TRISB2 = 1;			// RB2 ͂ɐݒ
	TRISBbits.TRISB7 = 0;			// RB7 o͂ɐݒ
	U1MODE = 0b1000100000001000;	//
	// PIC24FV32KA301 ́Cobt@tɎgƑMȂ悤B
	// ΍ƂāCFIFO ̍Ō𑗏oƂ̂݁C̃f[^ FIFO ɏނ悤ɂB 
	U1STA =  0b1000010000000000;	//
	U1BRG = 103;					// 38400bps
	//U1BRG = 416;					// 9600bps
	IEC0bits.U1RXIE = 1;			// ML
	IEC0bits.U1TXIE = 0;			// M
	IPC2bits.U1RXIP = 1;			// Dx = 1 (OS Ǘ̊)
	IPC3bits.U1TXIP = 1;			// Dx = 1 (OS Ǘ̊)

#if 0

	TRISBbits.TRISB8 = 1;			// RB8 ͂ɐݒ
	TRISBbits.TRISB7 = 0;			// RB7 o͂ɐݒ
	
	RPINR19bits.U2RXR = 8;			// U2RX  RP8 (RB8) ɐݒ
	RPOR3bits.RP7R = 5;				// U2TX  RP7 (RB7) ɐݒ

	U2MODE = 0b1000100010001000;	// 
	U2STA =  0b0000010000000000;	//
	U2BRG = 103;					// 38400bps
	IEC1bits.U2RXIE = 1;			// ML
	IEC1bits.U2TXIE = 0;			// M
	IPC7bits.U2RXIP = 1;			// Dx = 1 (OS Ǘ̊)
	IPC7bits.U2TXIP = 1;			// Dx = 1 (OS Ǘ̊)
#endif
	
	sDevUART.d_read = uart_read;
	sDevUART.d_write = uart_write;
	sDevUART.d_avail = uart_avail;
	sDevUART.d_chkready = uart_chkready;
	
	devUART = &sDevUART;
	
	RBTOS_Event_create(&sDevUART.evtDevice);
}

/**
 * U1RX .
 * nh U1RXInterrupt() R[.
 * @return
 *   AɃANeBuɂȂXbh.
 *   NULL ̏ꍇCXbh͐؂ւȂ.
 */
void U1RXInterruptFunc(void)
{
	IFS0bits.U1RXIF = 0;
	if (U1STAbits.OERR != 0) {
		U1STAbits.OERR = 0;
	}
	// Mobt@̋󂫃oCg̎Zo.
	short iv = nRxRD - nRxWR - 1;
	if (iv < 0) {
		iv += RXBUFFSIZE;
	}
	while (U1STAbits.URXDA != 0) {
		char ch = U1RXREG;
		if (iv != 0) {
			// Mobt@ɋ󂫂ȂΓǂݎ̂.
			rxBuff[nRxWR] = ch;
			nRxWR = (nRxWR + 1) & (RXBUFFSIZE - 1);
			iv--;
		}
	}
	iRBTOS_Event_set(&evtRX);
}


/**
 * U2TX .
 * nh U1TXInterrupt() R[.
 * @return
 *   AɃANeBuɂȂXbh.
 *   NULL ̏ꍇCXbh͐؂ւȂ.
 */
void U1TXInterruptFunc(void)
{
	if (nTxWR != nTxRD) {
		IFS0bits.U1TXIF = 0;
		// Mobt@̏[UoCǧvZ.
		short iv = nTxWR - nTxRD;
		if (iv < 0) {
			iv += TXBUFFSIZE;
		}
		// PIC24FV32KA301 ΍
		// FIFO ̍Ō𑗏oƂ̂݁C̃f[^ FIFO ɏނ悤ɂB 
		// ̂ɁCwhile łȂ if ƂB
		//while ((iv != 0) && (U2STAbits.UTXBF == 0)) {
		if ((iv != 0) && (U1STAbits.UTXBF == 0)) {
			// Mobt@Ƀf[^ȂCTX FIFO ɏ
			U1TXREG = txBuff[nTxRD];
			nTxRD = (nTxRD + 1) & (TXBUFFSIZE - 1);
			iv--;
		}
	} else {
		// Mobt@Ƀf[^ȂȂCM~
		IEC0bits.U1TXIE = 0;
	}
	iRBTOS_Event_set(&evtTX);
}
