/* vim:set ts=4 sts=4 sw=4 noet fenc=utf-8:

   Copyright 2009 senju@users.sourceforge.jp

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */

package jp.sourceforge.rabbitBTS.interceptors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.sourceforge.rabbitBTS.Sht;
import jp.sourceforge.rabbitBTS.controllers.IPublicController;
import jp.sourceforge.rabbitBTS.exceptions.HasNotValidIdException;
import jp.sourceforge.rabbitBTS.models.Account;
import jp.sourceforge.rabbitBTS.services.AccountService;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

/**
 * 認証チェック用インターセプタ
 */
public class AuthenticationInterceptor extends HandlerInterceptorAdapter {

	@Autowired
	private AccountService accountService;

	/**
	 * 認証チェックを行う。
	 * 
	 * @return ページが表示可能ならtrue、そうでないならfalse
	 */
	@Override
	public boolean preHandle(HttpServletRequest request,
			HttpServletResponse response, Object handler) throws Exception {
		try {
			final Account acc = this.accountService.fetchAccount();

			if (handler instanceof IPublicController || acc != null) {
				// 公開ページの場合、登録してなくても戻り値はtrue
				return true;
			}
			// アカウント未登録の場合、登録ページにリダイレクトする。
			response.sendRedirect("/register/");
			Sht.log(this).warn(
					"403(not registerd.) {}", request.getRequestURL());
			return false;
		} catch (final HasNotValidIdException e) {
			if (handler instanceof IPublicController) {
				// 公開ページの場合、ログインしてなくても戻り値はtrue
				return true;
			}

			// ログインしていない場合、ログインページにリダイレクトする。
			response.sendRedirect(Sht.loginUrl(request));
			Sht.log(this).warn("403 {}", request.getRequestURL());
			return false;
		}
	}

	/**
	 * viewでの描画用にアカウント、ログインURL、ログアウトURLをセットする。
	 */
	@Override
	public void postHandle(HttpServletRequest request,
			HttpServletResponse response, Object handler, ModelAndView mav)
			throws Exception {
		if (mav != null) {
			// リダイレクトする場合は不要
			if (!StringUtils.startsWith(mav.getViewName(), "redirect:")) {
				mav.addObject("acc", this.accountService.getCurrentAccount());
				mav.addObject("loginurl", Sht.loginUrl(request));
				mav.addObject("logouturl", Sht.logoutUrl(request));
			}
		}
	}

}
