// TTAOutputer.cs
// 2008/12/24

using System;
using System.IO;
using System.Text;
using System.Collections.Generic;

namespace QAX {

// TTAOutputer
public class TTAOutputer {

	private TTA.Reader _reader;

	private UInt64 _size;
	private UInt64 _samples;

	private Int32[] _pidx;

	public TTAOutputer(Stream s)
	{
		_reader = new TTA.Reader(s);

		_size = 0;

		_samples = 0;
	}

	public UInt64 Size {
		get { return _size; }
	}

	public UInt64 Samples {
		get { return _samples; }
	}

	public UInt64 OutputContent(Stream os)
	{
		_reader.ReadHeader();

		Int32 frame_no = 0;

		_size = 0;

		_samples = _reader.Samples;

		var idxs = new List<Int32>();

		for (; ; ) {
			Byte[] frame = _reader.FetchFrameData();
			if (frame == null) {
				break;
			}

			if (frame.Length < 4) {
				throw new TTA.FormatError("Frame.Size");
			}

			Int32 packet_size = frame.Length - 4;

			UInt32 fcrc = BitConverter.ToUInt32(frame, packet_size);
			{
				var crc = new QVorbis.CRC32();
				if (crc.Generate(frame, 0, packet_size) != fcrc) {
					throw new TTA.FormatError("Frame.CRC");
				}
			}

			Byte flags = 0;
			if (frame_no == 0) {
				flags |= QAXWriter.F_START;
			}

			if (frame_no == _reader.Frames - 1) {
				flags |= QAXWriter.F_END;
			}

			var ms = new MemoryStream();

			using (var w = new BinaryWriter(ms)) {
				w.Write(flags);

				Utils.WriteArray(w, new Int32[] { packet_size });

				w.Write(frame, 0, packet_size);
			}

			Byte[] page = ms.ToArray();

			Byte[] bcrc;
			{
				var crc = new QVorbis.CRC32();
				bcrc = BitConverter.GetBytes(
					crc.Generate(page));
			}

			os.Write(page, 0, page.Length);
			os.Write(bcrc, 0, bcrc.Length);

			Int32 page_size = page.Length + bcrc.Length;

			_size += (UInt64)page_size;

			frame_no += 1;

			idxs.Add(page_size);
		}

		UInt64 tsize = QAXContentWriter.Finish(os, _size, QAXWriter.PAGE_UNIT);

		_pidx = idxs.ToArray();

		return tsize;
	}

	public Byte[] GetSetup()
	{
		var ms = new MemoryStream();

		using (var w = new BinaryWriter(ms)) {
			Byte[] sig = new Byte[] { (Byte)'T', (Byte)'T', (Byte)'A', (Byte)'1' };

			w.Write(sig);

			w.Write(_reader.Channels);
			w.Write(_reader.BitsPerSample);
			w.Write(_reader.SamplingRate);
			w.Write(_reader.SamplesPerFrame);
		}

		return ms.ToArray();
	}

	public Byte[] GetIndex()
	{
		var ms = new MemoryStream();

		using (var w = new BinaryWriter(ms)) {
			Utils.WriteArray(w, new Int32[] { _pidx.Length });

			Utils.WriteArray(w, _pidx);
		}

		return ms.ToArray();
	}

} // TTAOutputer

} // namespace QAX

