// QStreamer.h
// 2009/08/21

#pragma once

/* */

#ifndef STDCALL
#define STDCALL __stdcall
#endif

/* */

// QEvent
class QEvent {

	HANDLE m_hEvent;

public:

	QEvent() : m_hEvent(0)
	{
	}

	~QEvent()
	{
		Close();
	}

	operator HANDLE()
	{
		return m_hEvent;
	}

	HRESULT Create(bool init)
	{
		HANDLE h = CreateEventW(
			0,
			TRUE,
			init,
			0);
		if (h == 0) {
			DWORD error = GetLastError();
			return HRESULT_FROM_WIN32(error);
		}

		m_hEvent = h;

		return S_OK;
	}

	void Close()
	{
		if (m_hEvent != 0) {
			CloseHandle(m_hEvent);
			m_hEvent = 0;
		}
	}

	void SetEvent()
	{
		::SetEvent(m_hEvent);
	}

	void ResetEvent()
	{
		::ResetEvent(m_hEvent);
	}

}; // QEvent

/* */

#define WM_STREAM_START (WM_USER + 100)

// QStreamer
class QStreamer {

	QReader* m_pReader;

	DWORD  m_idThread;
	HANDLE m_hThread;

	QEvent m_Ready;

	ATL::CComCriticalSection m_cs;

	bool  m_bSeek;
	INT64 m_SeekPos;

	ATL::CComPtr<IPin>          m_AudioPin;
	ATL::CComPtr<IMemInputPin>  m_AudioInputPin;
	ATL::CComPtr<IMemAllocator> m_AudioAllocator;

	QWaveOutputer m_Outputer;

public:

	QStreamer(QReader* p) : m_pReader(p), m_idThread(0), m_hThread(0), m_bSeek(false), m_SeekPos(0)
	{
	}

	~QStreamer()
	{
		Uninitialize();
	}

	HRESULT Initialize()
	{
		HRESULT hRslt = m_Ready.Create(false);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		hRslt = m_cs.Init();
		if (FAILED(hRslt)) {
			return hRslt;
		}

		return S_OK;
	}

	void Uninitialize()
	{
		if (m_hThread != 0) {
			Join();

			CloseHandle(m_hThread);

			m_idThread = 0;
			m_hThread  = 0;
		}

		m_Ready.Close();

		m_cs.Term();
	}

	HRESULT Create()
	{
		unsigned tid = 0;
		uintptr_t th = _beginthreadex(
			0,
			0,
			ThreadProc,
			this,
			0,
			&tid);
		if (th == 0) {
			return E_OUTOFMEMORY;
		}

		HANDLE aEvents[2] = {
			(HANDLE)th,
			m_Ready
		};

		DWORD result = WaitForMultipleObjects(
			sizeof(aEvents) / sizeof(aEvents[0]),
			aEvents,
			FALSE,
			60 * 1000);
		if (result != WAIT_OBJECT_0 + 1) {
			CloseHandle((HANDLE)th);
			return E_FAIL;
		}

		m_idThread = (DWORD) tid;
		m_hThread  = (HANDLE)th;

		return S_OK;
	}

	INT32 Join()
	{
		DWORD code = DWORD(-1);

		if (m_hThread != 0) {
			WaitForSingleObject(m_hThread, INFINITE);

			GetExitCodeThread(m_hThread, &code);
		}

		return (INT32)code;
	}

	/* */

	void RequestStart()
	{
		PostThreadMessageW(
			m_idThread,
			WM_STREAM_START,
			0,
			0);
	}

	void RequestQuit()
	{
		PostThreadMessageW(
			m_idThread,
			WM_QUIT,
			0,
			0);
	}

	/* */

	HRESULT SetupAudio(
		IMemInputPin*       pin,
		IMemAllocator*      alloc,
		const WAVEFORMATEX* w)
	{
		if (pin == 0) {
			return S_OK;
		}

		ATL::CComQIPtr<IPin> vp(pin);
		if (vp == 0) {
			return E_NOINTERFACE;
		}

		m_AudioPin       = vp;
		m_AudioInputPin  = pin;
		m_AudioAllocator = alloc;

		m_Outputer.Setup(
			w->nChannels,
			w->nSamplesPerSec);

		return S_OK;
	}

	/* */

	HRESULT Commit()
	{
		if (m_AudioAllocator != 0) {
			HRESULT hRslt = m_AudioAllocator->Commit();
			if (FAILED(hRslt)) {
				return hRslt;
			}
		}

		return S_OK;
	}

	HRESULT Decommit()
	{
		if (m_AudioAllocator != 0) {
			HRESULT hRslt = m_AudioAllocator->Decommit();
			if (FAILED(hRslt)) {
				return hRslt;
			}
		}

		return S_OK;
	}

	/* */

	HRESULT BeginFlush()
	{
		if (m_AudioPin != 0) {
			m_AudioPin->BeginFlush();
		}

		return S_OK;
	}

	/* */

	void RequestSeek(INT64 pos)
	{
		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_cs);

		m_bSeek   = true;
		m_SeekPos = pos;
	}

	/* */

private:

	/* */

	static void SetupThreadName(const char* p)
	{
		INT_PTR para[4];

		para[0] = 0x1000;
		para[1] = (INT_PTR)p;
		para[2] = -1;
		para[3] = 0;

		__try {
			RaiseException(
				0x406D1388,
				0,
				sizeof(para) / sizeof(ULONG_PTR),
				(const ULONG_PTR*)para);

		} __except(EXCEPTION_CONTINUE_EXECUTION) {

		}
	}

	static unsigned STDCALL ThreadProc(void* p)
	{
		QStreamer* pThis = static_cast<QStreamer*>(p);

		unsigned code = pThis->OnThread();

		return code;
	}

	unsigned OnThread()
	{
		SetupThreadName("QStreamer");

		HRESULT hRslt = CoInitializeEx(0, COINIT_MULTITHREADED);
		if (FAILED(hRslt)) {
			return -1;
		}

		unsigned code = -1;

		if (Init()) {
			DoProcess();
			code = 0;
		}

		Uninit();

		CoUninitialize();

		return code;
	}

	bool Init()
	{
		ATLTRACE("QStreamer::Init()\n");

		// Prepare Windows Message Queue
		{
			MSG msg = { 0 };
			BOOL fPeek = PeekMessageW(
				&msg,
				0,
				0,
				0,
				PM_NOREMOVE);
		}

		m_Ready.SetEvent();

		return true;
	}

	void Uninit()
	{
		ATLTRACE("QStreamer::Uninit()\n");
	}

	void DoProcess()
	{
		for (; ; ) {
			MSG msg = { 0 };

			BOOL fPeek = PeekMessageW(
				&msg,
				0,
				0,
				0,
				PM_REMOVE);
			if (fPeek) {
				switch (msg.message) {
				case WM_STREAM_START:
					if (DoStreaming()) {
						return;
					}
					break;

				case WM_QUIT:
					return;
				}

			} else {
				WaitMessage();
			}
		}
	}

	bool DoStreaming()
	{
		ATLTRACE("QStreamer::DoStreaming() Start...\n");

		for (; ; ) {
			MSG msg = { 0 };

			BOOL fPeek = PeekMessageW(
				&msg,
				0,
				0,
				0,
				PM_REMOVE);
			if (fPeek) {
				if (msg.message == WM_QUIT) {
					m_Outputer.Flush();
					return true;
				}

			} else {
				{
					ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_cs);

					if (m_bSeek) {
						m_bSeek = false;

						m_Outputer.Flush();

						if (!m_pReader->Seek(m_SeekPos)) {
							ATLTRACE("QStreamer::DoStreaming() Seek ERROR!\n");
							break;
						}

						if (m_AudioPin != 0) {
							m_AudioPin->EndFlush();
						}
					}
				}

				INT32 code = m_pReader->Read(
					m_AudioAllocator,
					m_AudioInputPin,
					&m_Outputer);
				if (code < 0) {
					ATLTRACE("QStreamer::DoStreaming() Read ERROR!\n");
					break;
				}
				if (code == 0) {
					break;
				}
			}
		}

		m_Outputer.EndOfStream(m_AudioInputPin);

		if (m_AudioPin != 0) {
			m_AudioPin->EndOfStream();
		}

		ATLTRACE("QStreamer::DoStreaming() Finished.\n");

		return false;
	}

}; // QStreamer

