// QReader.h
// 2009/08/21

#pragma once

/* */

// QReader
class QReader {

public:

	struct Format {
		INT32 SamplingRate;
		INT32 Channels;
	};

	virtual ~QReader()
	{
	}

	virtual Format GetFormat() = 0;

	virtual INT64 GetDuration() = 0;

	virtual bool Seek(INT64 pos) = 0;

	virtual INT32 Read(
		IMemAllocator* aalloc,
		IMemInputPin*  apin,
		QWaveOutputer* aout) = 0;

	virtual void SetupLoopPoint(INT64 pos) = 0;

}; // QReader

const INT64 RT_UNIT = 10*1000*1000;

/* */

// QVorbisReader
class QVorbisReader : public QReader {

	QOV_Reader_t* m_reader;

	INT64 m_Position;

	INT64 m_Seek;

	enum {
		BUFFER_SIZE = 2048
	};

public:

	QVorbisReader() : m_reader(0), m_Position(0), m_Seek(-1)
	{
	}

	virtual ~QVorbisReader()
	{
		QOV_ReleaseReader(m_reader);
	}

	bool Open(const WCHAR* path)
	{
		m_reader = QOV_CreateReader();
		if (m_reader == 0) {
			return false;
		}

		if (!QOV_OpenReader(m_reader, path)) {
			return false;
		}

		return true;
	}

	bool Open(IStream* p)
	{
		m_reader = QOV_CreateReader();
		if (m_reader == 0) {
			return false;
		}

		if (!QOV_OpenReader_IStream(m_reader, p)) {
			return false;
		}

		return true;
	}

	/* */

	virtual Format GetFormat()
	{
		const QOV_Format_t* fmt = QOV_GetFormat(m_reader);
		Format f = { fmt->SamplingRate, fmt->Channels };
		return f;
	}

	virtual INT64 GetDuration()
	{
		const QOV_Format_t* fmt = QOV_GetFormat(m_reader);
		return (fmt->Duration * RT_UNIT) / fmt->SamplingRate;
	}

	/* */

	virtual bool Seek(INT64 pos)
	{
		const QOV_Format_t* fmt = QOV_GetFormat(m_reader);
		INT64 sample = (pos * fmt->SamplingRate) / RT_UNIT;

		if (!QOV_Seek(m_reader, sample)) {
			return false;
		}

		m_Position = 0;

		return true;
	}

	virtual INT32 Read(
		IMemAllocator* aalloc,
		IMemInputPin*  apin,
		QWaveOutputer* aout)
	{
		INT16 buffer[BUFFER_SIZE * 2];

		INT32 output = 0;
		while (output == 0) {
			if (!QOV_Decode(m_reader, buffer, BUFFER_SIZE, &output)) {
				return -1;
			}
			if (output == 0) {
				if (m_Seek < 0) {
					return 0;
				}

				if (!QOV_Seek(m_reader, m_Seek)) {
					return -1;
				}
			}
		}

		const QOV_Format_t* fmt = QOV_GetFormat(m_reader);

		REFERENCE_TIME rts = (m_Position * RT_UNIT) / fmt->SamplingRate;

		if (!aout->Output(
			aalloc,
			apin,
			rts,
			buffer,
			output * fmt->Channels)) {
			return -1;
		}

		m_Position += output;

		return 1;
	}

	virtual void SetupLoopPoint(INT64 pos)
	{
		m_Seek = pos;
	}

}; // QVorbisReader

/* */

// QTTAReader
class QTTAReader : public QReader {

	QTTA_Reader_t* m_reader;

	INT64 m_Position;

	INT64 m_Seek;

	enum {
		BUFFER_SIZE = 2048
	};

public:

	QTTAReader() : m_reader(0), m_Position(0), m_Seek(-1)
	{
	}

	virtual ~QTTAReader()
	{
		QTTA_ReleaseReader(m_reader);
	}

	bool Open(const WCHAR* path)
	{
		m_reader = QTTA_CreateReader();
		if (m_reader == 0) {
			return false;
		}

		if (!QTTA_OpenReader(m_reader, path)) {
			return false;
		}

		return true;
	}

	bool Open(IStream* p)
	{
		m_reader = QTTA_CreateReader();
		if (m_reader == 0) {
			return false;
		}

		if (!QTTA_OpenReader_IStream(m_reader, p)) {
			return false;
		}

		return true;
	}

	/* */

	virtual Format GetFormat()
	{
		const QTTA_Format_t* fmt = QTTA_GetFormat(m_reader);
		Format f = { fmt->SamplingRate, fmt->Channels };
		return f;
	}

	virtual INT64 GetDuration()
	{
		const QTTA_Format_t* fmt = QTTA_GetFormat(m_reader);
		return (fmt->Duration * RT_UNIT) / fmt->SamplingRate;
	}

	/* */

	virtual bool Seek(INT64 pos)
	{
		const QTTA_Format_t* fmt = QTTA_GetFormat(m_reader);
		INT64 sample = (pos * fmt->SamplingRate) / RT_UNIT;

		if (!QTTA_Seek(m_reader, sample)) {
			return false;
		}

		m_Position = 0;

		return true;
	}

	virtual INT32 Read(
		IMemAllocator* aalloc,
		IMemInputPin*  apin,
		QWaveOutputer* aout)
	{
		INT16 buffer[BUFFER_SIZE * 2];

		INT32 output = 0;
		while (output == 0) {
			if (!QTTA_Decode(m_reader, buffer, BUFFER_SIZE, &output)) {
				return -1;
			}
			if (output == 0) {
				if (m_Seek < 0) {
					return 0;
				}

				if (!QTTA_Seek(m_reader, m_Seek)) {
					return -1;
				}
			}
		}

		const QTTA_Format_t* fmt = QTTA_GetFormat(m_reader);

		REFERENCE_TIME rts = (m_Position * RT_UNIT) / fmt->SamplingRate;

		if (!aout->Output(
			aalloc,
			apin,
			rts,
			buffer,
			output * fmt->Channels)) {
			return -1;
		}

		m_Position += output;

		return 1;
	}

	virtual void SetupLoopPoint(INT64 pos)
	{
		m_Seek = pos;
	}

}; // QTTAReader

/* */

