/* OggReader.c */
/* 2009/07/22  */

#include "StdAfx.h"

#include "OggReader.h"

/* */

INT32 QO_ScanCapturePattern(
	const VOID* pv,
	SIZE_T      cb)
{
	                          /* (3)   2   (1)   0  */
	static const UINT8 CP[4] = { 'O', 'g', 'g', 'S' };

	const UINT8* p = (const UINT8*)pv;
	const UINT8* e = p + cb;

	p += 3;

	while (p < e) {
		SIZE_T i;
		for (i = 3; *p == CP[i]; p--, i--) {
			if (i == 0) {
				return p - (const UINT8*)pv;
			}
		}

		if (*p == 'g') {
			p += 4 - i; /* 1, 2, 3 */
		} else if (*p == 'O') {
			p += 3;
		} else {
			p += 4;
		}
	}

	return -1;
}

/* */

#define OR_BUFFER_SIZE  0x10000
#define OR_PACKETS_SIZE 0x10000
#define OR_PAYLOAD_SIZE  0x4000

BOOL QO_OggReader_Init(
	QO_OggReader_t*    t,
	QO_MemoryPool_t*   pool,
	QO_Allocator_t*    allocator,
	QO_StreamReader_t* reader)
{
	memset(t, 0, sizeof(QO_OggReader_t));

	t->Reader = reader;

	t->Buffer = (UINT8*)QO_MemoryPool_Allocate(pool, allocator, OR_BUFFER_SIZE);
	if (t->Buffer == NULL) {
		return FALSE;
	}

	t->Size = OR_BUFFER_SIZE;

	t->Pointer = 0;
	t->Filled  = 0;

	t->Header = (UINT8*)QO_MemoryPool_Allocate(pool, allocator, 0x200);
	if (t->Header == NULL) {
		return FALSE;
	}

	t->Packets = 0;

	t->PacketsSize = (SIZE_T*)QO_MemoryPool_Allocate(pool, allocator, sizeof(SIZE_T) * 0x100);
	if (t->PacketsSize == NULL) {
		return FALSE;
	}

	t->PacketsBuffer = (UINT8*)QO_MemoryPool_Allocate(pool, allocator, OR_PACKETS_SIZE);
	if (t->PacketsBuffer == NULL) {
		return FALSE;
	}

	t->Discontinue = FALSE;

	t->Payload = (UINT8*)QO_MemoryPool_Allocate(pool, allocator, OR_PAYLOAD_SIZE);
	if (t->Payload == NULL) {
		return FALSE;
	}

	t->PayloadSize = 0;

	return TRUE;
}

/* */

static BOOL ReadNext(
	QO_OggReader_t* t)
{
	SIZE_T next = (t->Pointer + t->Filled) & (t->Size - 1);
	SIZE_T size = t->Size - t->Filled, cb;
	if (size > t->Size / 2) {
		size = t->Size / 2;
	}

	cb = t->Reader->Read(t->Reader->Context, t->Buffer + next, size);
	if (cb == 0) {
		return FALSE;
	}

	t->Filled += cb;

	return TRUE;
}

static BOOL ReadBuffer(
	QO_OggReader_t* t,
	VOID*           pv,
	SIZE_T          cb)
{
	UINT8* p = (UINT8*)pv;
	UINT8* e = p + cb;

	if (t->Filled < cb) {
		if (!ReadNext(t)) {
			return FALSE;
		}

		if (t->Filled < cb) {
			return FALSE;
		}
	}

	while (p < e) {
		*(p++) = t->Buffer[t->Pointer];
		t->Pointer = (t->Pointer + 1) & (t->Size - 1);
	}

	t->Filled -= cb;

	return TRUE;
}

static BOOL SkipBuffer(
	QO_OggReader_t* t,
	SIZE_T          cb)
{
	if (t->Filled < cb) {
		if (!ReadNext(t)) {
			return FALSE;
		}

		if (t->Filled < cb) {
			return FALSE;
		}
	}

	t->Pointer = (t->Pointer + cb) & (t->Size - 1);
	t->Filled -= cb;

	return TRUE;
}

/* */

/* Ogg Header

+ 0 CapturePattern
+ 3

+ 4 Version

+ 5 HeaderType

+ 6 GranulePosition
+13

+14 StreamSerialNumber
+17

+18 PageSequenceNumber
+21

+22 CRC32
+25

+26 Segments

+27+

*/

/* */

static BOOL ReadHeader(
	QO_OggReader_t* t)
{
	SIZE_T SZ[2] = { 27 };

	UINT8* pb = t->Header;

	SIZE_T i;
	for (i = 0; i < 2; i++) {
		if (!ReadBuffer(t, pb, SZ[i])) {
			return FALSE;
		}

		pb += SZ[i];

		if (i == 0) {
			if (memcmp(t->Header, "OggS\0", 5) != 0) {
				return FALSE;
			}

			SZ[1] = pb[-1]; /* Segments */
		}
	}

	return TRUE;
}

/* */

#define TO32(X) ((t->Header[X+3] << 24) | (t->Header[X+2] << 16) | (t->Header[X+1] << 8) | t->Header[X+0])

BOOL QO_OggReader_ReadHeader(
	QO_OggReader_t* t,
	QO_OggHeader_t* header)
{
	SIZE_T idx = 0, len = 0;
	UINT8* p = t->Header + 27;
	UINT8* e;
	SIZE_T* sz = t->PacketsSize;

	if (!ReadHeader(t)) {
		return FALSE;
	}

	header->HeaderType = t->Header[5];

	header->GranulePosition = (((UINT64)TO32(10)) << 32) | TO32(6);

	header->StreamSerialNumber = TO32(14);
	header->PageSequenceNumber = TO32(18);

	header->CRC32 = TO32(22);

	for (e = p + t->Header[26]; p < e; p++) {
		len += *p;

		if (*p < 0xff) {
			sz[idx++] = len;
			len       = 0;
		}
	}

	sz[idx] = len;

	t->Packets = idx;

	return TRUE;
}

#undef TO32

/* */

#define HT_CONT 0x01

BOOL QO_OggReader_ReadPayload(
	QO_OggReader_t* t)
{
	INT32 i;

	UINT8* pb = t->PacketsBuffer;

	t->Discontinue = FALSE;

	if (t->Packets > 0 && (t->Header[5] & HT_CONT) != 0) {
		if (t->PayloadSize > 0) {
			memcpy(pb, t->Payload, t->PayloadSize);
			pb += t->PayloadSize;

		} else {
			t->Discontinue = TRUE;
		}
	}

	for (i = 0; i < t->Packets; i++) {
		if (!ReadBuffer(t, pb, t->PacketsSize[i])) {
			return FALSE;
		}
		pb += t->PacketsSize[i];
	}

	if (t->Packets > 0 && (t->Header[5] & HT_CONT) != 0) {
		t->PacketsSize[0] += t->PayloadSize;
		t->PayloadSize = 0;
	}

	if (t->PacketsSize[i] > 0) {
		if (!ReadBuffer(t, t->Payload + t->PayloadSize, t->PacketsSize[i])) {
			return FALSE;
		}

		t->PayloadSize += t->PacketsSize[i];
	}

	return TRUE;
}

/* */

BOOL QO_OggReader_SkipPayload(
	QO_OggReader_t* t)
{
	INT32 i;

	t->Discontinue = TRUE;

	for (i = 0; i < t->Packets; i++) {
		if (!SkipBuffer(t, t->PacketsSize[i])) {
			return FALSE;
		}
	}

	if (t->PacketsSize[i] > 0) {
		if (!SkipBuffer(t, t->PacketsSize[i])) {
			return FALSE;
		}
	}

	return TRUE;
}

/* */

BOOL QO_OggReader_Seek(
	QO_OggReader_t* t,
	INT64           pos)
{
	INT32 s;

	if (!t->Reader->Seek(t->Reader->Context, pos)) {
		return FALSE;
	}

	t->Pointer = 0;
	t->Filled  = 0;

	if (!ReadNext(t)) {
		return FALSE;
	}

	s = QO_ScanCapturePattern(t->Buffer, t->Filled);
	if (s < 0) {
		return FALSE;
	}

	t->Pointer  = s;
	t->Filled  -= s;

	t->Packets     = 0;
	t->PayloadSize = 0;

	return TRUE;
}

/* */

