// QASDecoder.h
// 2008/12/26

#pragma once

namespace QAX {

// QASDecoder
class QASDecoder {

	QASHeader m_Header;

	CHAR m_Format[16];

	BYTE*  m_MetaData;
	SIZE_T m_MetaSize;

	const BYTE* m_Setup;
	const BYTE* m_SetupEnd;

	const BYTE* m_IndexStart;
	const BYTE* m_IndexEnd;

	enum {
		ID_VORBIS,
		ID_TTA1
	};

public:

	QASDecoder() :
		m_MetaData(0),
		m_MetaSize(0),
		m_Setup(0),
		m_SetupEnd(0),
		m_IndexStart(0),
		m_IndexEnd(0)
	{
		ZeroMemory(&m_Header, sizeof(m_Header));
		ZeroMemory(&m_Format, sizeof(m_Format));
	}

	~QASDecoder()
	{
		if (m_MetaData != 0) {
			free(m_MetaData);
			m_MetaData = 0;
		}
	}

	VOID* DetachMetaData()
	{
		VOID* p = m_MetaData;
		m_MetaData = 0;
		return p;
	}

	QDecoderOutput* CreateOutput(
		QDecoderFactory* pFactory,
		IStream*         pStream)
	{
		SetupHeader(pStream);

		INT32 id = -1;
		if (strcmp(m_Format, "vorbis") == 0) {
			id = ID_VORBIS;
		} else if (strcmp(m_Format, "TTA1") == 0) {
			id = ID_TTA1;
		} else {
			return 0;
		}

		SetupMetaData(pStream);

		QDecoderOutput* output = 0;

		switch (id) {
		case ID_VORBIS:
		{
			QASVorbisOutputImpl* impl = 0;

			try {
				impl = new QASVorbisOutputImpl();

				impl->SetupHeader(&m_Header);

				impl->SetupMetaData(
					DetachMetaData(),
					m_Setup,
					m_SetupEnd,
					m_IndexStart,
					m_IndexEnd);

				impl->SetupDecoder(pFactory);

				impl->Prepare(pStream);

				output = impl;

			} catch (...) {
				if (impl != 0) {
					impl->Release();
				}
				throw;
			}

			break;
		}

		case ID_TTA1:
		{
			QASTTAOutputImpl* impl = 0;

			try {
				impl = new QASTTAOutputImpl();

				impl->SetupDecoder(
					m_Setup,
					m_SetupEnd - m_Setup,
					m_Header.Samples);

				StreamReader* reader = IStream_Reader::Create(
					pStream,
					m_Header.ContentPos);

				impl->SetupReader(
					reader,
					SIZE_T(m_Header.ContentSize));

				impl->SetupIndex(
					DetachMetaData(),
					m_IndexStart,
					m_IndexEnd - m_IndexStart);

				output = impl;

			} catch (...) {
				if (impl != 0) {
					impl->Release();
				}
				throw;
			}

			break;
		}

		} // switch

		return output;
	}

private:

	void SetupHeader(IStream* pStream)
	{
		ULONG cb = 0;
		HRESULT hRslt = pStream->Read(
			&m_Header,
			sizeof(m_Header),
			&cb);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Read", hRslt);
		}

		if (cb != sizeof(m_Header)) {
			FormatError::Throw("QASHeader.EOS");
		}

		if (memcmp(m_Header.Signature, "QAS1", 4) != 0) {
			FormatError::Throw("QASHeader.Signature");
		}

		lstrcpynA(m_Format, (CHAR*)m_Header.Format, 12);
	}

	void SetupMetaData(IStream* pStream)
	{
		m_MetaData = static_cast<BYTE*>(malloc(m_Header.MetaDataSize));
		if (m_MetaData == 0) {
			DecoderError::Throw("OutOfMemory");
		}

		m_MetaSize = m_Header.MetaDataSize;

		LARGE_INTEGER pos;
		ULARGE_INTEGER newPos;

		pos.QuadPart = m_Header.MetaDataPos;
		HRESULT hRslt = pStream->Seek(
			pos,
			STREAM_SEEK_SET,
			&newPos);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Seek", hRslt);
		}

		ULONG cb = 0;
		hRslt = pStream->Read(
			m_MetaData,
			m_MetaSize,
			&cb);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Read", hRslt);
		}

		if (cb != m_MetaSize) {
			FormatError::Throw("MetaData.EOS");
		}

		const BYTE* index = m_MetaData + m_Header.MetaDataIndex;
		if (index < m_MetaData || index >= m_MetaData + m_MetaSize) {
			FormatError::Throw("MetaData.Index");
		}

		const BYTE* p   = index;
		const BYTE* end = m_MetaData + m_MetaSize;

		UINT32 count;
		p += DecodeVNumber(p, end, &count);

		if (count != 2) {
			FormatError::Throw("MetaData.Count");
		}

		UINT32 setup_len;
		p += DecodeVNumber(p, end, &setup_len);

		UINT32 index_len;
		p += DecodeVNumber(p, end, &index_len);

		m_Setup      = m_MetaData;
		m_SetupEnd   = m_MetaData + setup_len;
		m_IndexStart = m_SetupEnd;
		m_IndexEnd   = m_IndexStart + index_len;

		if (m_SetupEnd > m_MetaData + m_MetaSize) {
			FormatError::Throw("MetaData.Setup.Size");
		}

		if (m_IndexEnd > m_MetaData + m_MetaSize) {
			FormatError::Throw("MetaData.Index.Size");
		}
	}

}; // QASDecoder

} // namespace QAX

