#!/usr/bin/perl -- 
use utf8;
use strict;
use warnings;
use Encode;
use DBI;
use Digest::MD5  qw(md5 md5_hex md5_base64);
use Carp qw(cluck confess);
use POSIX qw(setsid strftime);
use Socket; # for inet_ntoa()
use POE qw(
	Wheel::SocketFactory
	Wheel::ReadWrite
	Driver::SysRW
	Filter::Line
	Component::Client::DNS
	Component::Client::TCP
);
use Getopt::Long;

our $conf_kill=0;
our $conf_file='proxy-config.pl';
our $utf8 = Encode::find_encoding('utf8');
our $now;
our $cookie;
our $form;
our $dbh;
our $param;
our $session;
our $verbose = 0;
our $sql_error;

sub reverseIPv4($){
	$_[0] =~ /(\d+)\.(\d+)\.(\d+)\.(\d+)/ or return;
	return "$4.$3.$2.$1.in-addr.arpa.";
}

# DNS resolver subprocess
# our $named = POE::Component::Client::DNS->spawn( Alias => "named" );

#####################################################################################
# low-level functions

# return first defined value in arguments list
sub dor{
	for(@_){ return $_ if defined $_; }
	return;
}

# make format time string
sub timestr{
	my($t) = @_;
	my @lt = localtime $t;
	$lt[5]+=1900;$lt[4]+=1;
	return sprintf "%d/%02d/%02d_%02d:%02d:%02d",reverse @lt[0..5];
}

# make format time string
sub timestr822{
	my($t) = @_;
	my ($week, $month, $day, $time, $year) = split /\s+/, scalar(localtime($t));
	return "$week, $day $month $year $time +0900";
}

# make size str
sub sizestr{
	my($s)=@_;
	( $s >= 1000000 ) and return sprintf("%.1fMB",$s/1000000);
	( $s >= 1000    ) and return sprintf("%.1fKB",$s/1000);
	return "${s}B";
}

# clip numeric in range
sub clipRange{
	return $_[0]<$_[1]?$_[1]:$_[0]>$_[2]?$_[2]:$_[0];
}

sub daemonize($){
	my($pidfile)=@_;
	# daemonize
	chdir '/'                 or die("Can't chdir to /: $!");
	umask 0;
	open STDIN, '/dev/null'   or die("Can't read /dev/null: $!");
	open STDOUT, '>/dev/null' or die("Can't write to /dev/null: $!");
	open STDERR, '>/dev/null' or die("Can't write to /dev/null: $!");
	defined(my $pid = fork)   or die("Can't fork: $!");
	if($pid){
		my $fh;
		open($fh,">",$pidfile) and print $fh $pid;
		exit;
	}
	setsid or die("Can't start a new session: $!");
}

#####################################################################################

sub getSQLResult{
	my $sql = shift;
	$sql_error = '';
	my $sth = $dbh->prepare($sql);
	if( not $sth ){
		$sql_error = $dbh->errstr;
		confess $sql_error,": sql=",$sql,"\n";
		return;
	}elsif( not $sth->execute( @_ ) ){
		$sql_error = $sth->errstr;
		cluck $sql_error,": sql=",$sql,"\n";
		$sth->finish;
		return;
	}
	defined(wantarray) and return $sth;
	$sth->finish;
}
sub getSQLRow{
	my $sth = getSQLResult(@_);
	$sth or return;
	my $h = $sth->fetchrow_hashref;
	$sth->finish;
	return $h;
}
sub doSQL{
	my $sth = getSQLResult(@_);
	my $r = ($sth?1:0);
	$sth and $sth->finish;
	return $r;
}

sub startTransaction(){
	for(my $try=0;$try<30;++$try){
		return if doSQL("begin immediate");
		cluck "begin : $sql_error\n";
		sleep 1;
	}
	exitError("lock failure at begin. $sql_error");
}
sub endTransaction(){
	for(my $try=0;$try<30;++$try){
		return if doSQL("commit");
		last if $sql_error =~ /statements in progress/;
		cluck "commit: $sql_error\n";
		sleep 1;
	}
	exitError("lock failure at commit. $sql_error");
}

sub db_update($$$$){
	my($table,$keyname,$key,$item)=@_;
	my @keys = keys %$item;
	my $sql = "update $table set ".join(',',map{ "${_}=?"} @keys )." where $keyname=?";
	return doSQL($sql,(map{$item->{$_}} @keys),$key );
}

sub db_insert($$){
	my($table,$item)=@_;
	my @keys = keys %$item;
	my $sql = "insert into $table(".join(',',@keys).")values(".join(',',map{'?'}@keys).")";
	doSQL($sql,map{$item->{$_}} @keys ) and return $dbh->func('last_insert_rowid');
	return; # return undef if failed
}

##################################################################

sub initDB{
	$dbh = DBI->connect("dbi:SQLite:dbname=$param->{sitedir}/data/.db.sqlite","","");

	# make db
	$dbh->do($_) for grep{length} split /\s*;\s*/,<<'END';

create table if not exists session(
	sid INTEGER PRIMARY KEY AUTOINCREMENT,
	uid text,
	lastaccess integer,
	sender text
);
create UNIQUE index if not exists session_uid on session(uid);
create index if not exists session_lastaccess on session(lastaccess);

create table if not exists command (
	cid INTEGER PRIMARY KEY AUTOINCREMENT,
	sid integer,
	line text
);
create index if not exists command_sid on command(sid,cid);

create table if not exists response (
	rid INTEGER PRIMARY KEY AUTOINCREMENT,
	sid integer,
	type text,
	extra text
);
create index if not exists response_sid on response(sid,rid);

END
}

###############################################################################

sub sendLine{
	my($heap,$line)=@_;
	$heap->{server}->put( $heap->{encoding}->encode( $line ),"\x0d\x0a");
}
sub sendCommand{
	my($heap,@args)=@_;
	if( @args >= 2 ){
		my $trail = pop @args;
		sendLine($heap,join(' ',@args)." :$trail");
	}else{
		sendLine($heap,join(' ',@args));
	}
}
sub saveLine{
	my($heap,$type,$extra)=@_;
	db_insert("response",{
		sid=>$heap->{sid},
		type=>$type,
		extra=>$extra,
	});
}

our %client_callback =(
	Started => sub{
		# start args
		my($kernel,$heap,@args) = @_[KERNEL,HEAP,ARG0..$#_];
		$heap->{sid}     = $args[0];
		$heap->{loghead} = "sid$heap->{sid}_cid".$_[SESSION]->ID;
		$heap->{encoding} = Encode::find_encoding($param->{irc_server_encoding});

		$verbose and warn  "$heap->{loghead} Started\n";
		# シグナルハンドラ
		$kernel->sig( INT  => 'signal_handler' );
		$kernel->sig( TERM => 'signal_handler' );
	},
	Connected => sub{
		my ($heap,$socket, $peer_address, $peer_port) = @_[HEAP,ARG0, ARG1, ARG2];
		$verbose and warn  "$heap->{loghead} Connected\n";
		saveLine($heap,"Connect","");
	},
	ConnectError => sub{
		my ($heap,$syscall_name, $error_number, $error_string) = @_[HEAP,ARG0, ARG1, ARG2];
		$verbose and warn  "$heap->{loghead} ConnectError $syscall_name:$error_string\n";
		saveLine($heap,"ConnectError","$syscall_name:$error_string");
	},
	Disconnected => sub {
		my ($heap) = $_[HEAP];
	    # no special parameters
		$verbose and warn  "$heap->{loghead} Disconnected\n";
		saveLine($heap,"Close","");
	},
	ServerInput => sub {
		my($heap,$line) = @_[HEAP,ARG0];
		$line = Encode::decode( $heap->{encoding},$line);
		$line =~ s/[\x0d\x0a]+$//;
		saveLine($heap,"SocketData",$line);
		# いくつかのメッセージは中継側で処理する
		if( $line =~ s/^PING(\s.*)/PONG$1/ ){
			 sendLine($heap,$line);
		}
	},
	ServerError => sub {
	    my ($heap,$syscall_name, $error_number, $error_string) = @_[HEAP,ARG0, ARG1, ARG2];
		warn "$heap->{loghead} ServerError $syscall_name:$error_string\n";
		saveLine($heap,"ServerError","$syscall_name:$error_string");
	},
	ServerFlushed => sub {
		my ($heap) = $_[HEAP];
	},
);

our $client_events = {
	# シグナル
	signal_handler => sub{
		my ($kernel,$heap, $signal) = @_[KERNEL,HEAP,ARG0];
		if( $signal eq 'HUP' ){
			# client connection ignore HUP signal
		}elsif( $heap->{server} ){
			warn "$heap->{loghead} signal $signal\n";
			delete $heap->{server};
		}
		$kernel->sig_handled;
	},
	command => sub{
		my($kernel,$heap,$line) = @_[KERNEL,HEAP,ARG0];
		if( $line eq "connect" ){
			$kernel->yield( "reconnect" );
			return;
		}
		if( $line eq "disconnect" ){
			 $kernel->yield( "shutdown" );
			return;
		}
		sendLine($heap,$line);
	},
	exit => sub{
		my($kernel,$heap) = @_[KERNEL,HEAP];
		$verbose and warn "$heap->{loghead} exit\n";
		delete $heap->{server};
	},
};

#########################################

our %session_map;

sub prepareIRCClient{
	my($sid)=@_;
	my $id = $session_map{$sid};
	if(not $id){
		$id = POE::Component::Client::TCP->new(
			RemoteAddress  => $param->{irc_server_host},
			RemotePort     => $param->{irc_server_port},
		#	BindAddress    => '0.0.0.0',
		#	BindPort       => 0,
			ConnectTimeout => 5,              # Seconds; optional.

			# filter
			Filter => POE::Filter::Line->new(),
			# events
			InlineStates   => $client_events,
			# callbacks
			%client_callback,

			# arguments for Started callback
			Args  => [ $sid], 
	    );
		if($id){
			$session_map{$sid}=$id;
			$poe_kernel->refcount_increment($id, 'db_session' );
		}

	}
	return $id;
}

sub closeIRCClient{
	my($sid)=@_;
	my $id = $session_map{$sid};
	if($id){
		$poe_kernel->refcount_decrement($id, 'db_session' );
		$poe_kernel->call($id,"exit");
		delete $session_map{$sid};
	}
}


sub update{
	startTransaction();
	# remove expired session
	{
		my $expire = time-60*2.5;
		my $sth = getSQLResult("select * from session where lastaccess<?",$expire);
		while( my $item = $sth->fetchrow_hashref ){
			my $sid = $item->{sid};
			$verbose and warn "six$sid expire\n";
			closeIRCClient( $sid );
			doSQL("delete from command where sid=?",$sid);
			doSQL("delete from response where sid=?",$sid);
		}
		$sth->finish;
		doSQL("delete from session where lastaccess<?",$expire);
	}
	# send command 
	{
		my $cid;
		my $sth = getSQLResult("select * from command order by cid");
		while( my $item = $sth->fetchrow_hashref ){
			$cid = $item->{cid};
			my $id = prepareIRCClient($item->{sid}) or next;
			my $line = $utf8 ->decode($item->{line});
			$poe_kernel->post($id,"command",$line);
		}
		$sth->finish;
		defined($cid) and doSQL("delete from command where cid<=?",$cid);
	}
	endTransaction();
}

sub openLog{
	open(STDERR,">>:encoding(utf8)","$param->{sitedir}/data/proxy-err.log");
}

our $timer_events ={
	_start => sub{
		my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
		$heap->{loghead} = "(timer)";
		$verbose and warn "$heap->{loghead} _start\n";
		# シグナルハンドラを登録
		$kernel->sig( INT  => 'signal_handler' );
		$kernel->sig( TERM => 'signal_handler' );
		$kernel->sig( HUP  => 'signal_handler' );
		# アラームを登録
		$heap->{next} = time + 1;
		$heap->{alarm_id} = $kernel->alarm_set( "alarm",$heap->{next} );
		# initialize
		openLog();
	},
	_stop => sub{
		my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
		$verbose and warn "$heap->{loghead} _stop\n";
	},
	# アラーム
	alarm => sub{
		my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
		$kernel->alarm_remove( $heap->{alarm_id} );
		$heap->{next} += 1;
		$heap->{alarm_id} = $kernel->alarm_set( "alarm",$heap->{next} );
		update();
	},
	# シグナル
	signal_handler => sub{
		my ($kernel, $signal) = @_[KERNEL, ARG0];
		$kernel->yield( "signal_lazy",$signal);
		$kernel->sig_handled;
	},
	# シグナル(yield後)
	signal_lazy => sub{
		my ( $kernel,$heap,$signal) = @_[KERNEL,HEAP, ARG0];
		warn "$heap->{loghead} signal $signal\n";
		if( $signal eq 'HUP' ){
			update();
		}else{
			$kernel->alarm_remove( $heap->{alarm_id} );
		}
	},
};

sub main{
	$now = time;

	GetOptions(
		"stop:+"=>\$conf_kill,
		"file=s"=>\$conf_file,
	) or die "Bad options\n";

	$param= eval{ do $conf_file;};
	$@ and die $@;
	chdir $param->{sitedir} or exitError("cannot chdir to sitedir:$!");

	my $pidfile = "$param->{sitedir}/data/ircproxy.pid";
	if( $conf_kill ){
		-f $pidfile or die "missing pid file: $pidfile\n";
		system("kill `cat $pidfile`");
		unlink $pidfile;
		exit;
	}
	-f $pidfile and die "already exists pidfile: $pidfile\n";

	daemonize($pidfile);
	openLog();
	initDB();
	doSQL("delete from command");
	doSQL("delete from response");

	# create timer session
	POE::Session->create( inline_states => $timer_events );
	warn "Running.\n";
	$poe_kernel->run();
}


main();
