#ifndef QRK_CYCLE_TIMER_H
#define QRK_CYCLE_TIMER_H

/*!
  \file
  \brief 周期タイマー

  \author Satofumi KAMIMURA

  $Id: CycleTimer.h 758 2009-04-21 12:45:57Z satofumi $
*/

#include <memory>


namespace qrk
{
  /*!
    \brief 周期タイマー
  */
  class CycleTimer
  {
    CycleTimer(const CycleTimer& rhs);
    CycleTimer& operator = (const CycleTimer& rhs);

    struct pImpl;
    std::auto_ptr<pImpl> pimpl;

  public:
    enum {
      DefaultFps = 60,          //!< 初期 FPS
    };

    CycleTimer(void);
    ~CycleTimer(void);


    /*!
      \brief 待機

      指定周期を実現する待機時間を返す

      \retval 待つべき時間 [msec]

      使用例
\dontinclude libs/gui/samples/CycleTimerSample.cpp
\skip CycleTimer cycle_timer;
\until ++total_count;
\line }
    */
    size_t waitMsec(void);


    /*!
      \brief タイマー動作のリセット

      周期の動作カウントをリセットする
    */
    void reset(void);


    /*!
      \brief 周期処理を必ず繰り返すか、の指定

      \param[in] on 周期処理を厳密に行う必要がある場合 true
    */
    void setStrictCycle(bool on);


    /*!
      \brief FPS による周期指定

      \param[in] fps 動作周期

      \see setCycleMsec()
    */
    void setCycleFps(size_t fps);


    /*!
      \brief 周期の取得

      \retval 動作周期 [fps]
    */
    size_t cycleFps(void) const;


    /*!
      \brief msec による周期指定

      \param[in] delay_msec 周期毎の間隔

      \see setCycleFps()
    */
    void setCycleMsec(size_t delay_msec);


    /*!
      \brief 周期の取得

      \retval 動作周期 [msec]
    */
    size_t cycleMsec(void) const;


    /*!
      \brief 次の周期までの時間を返す

      \retval 次の周期までの時間 [msec]
    */
    int toNextCycleMsec(void);
  };
}

#endif /* !QRK_CYCLE_TIMER_H */
