/*!
  \file
  \brief シリアル通信の実処理 (Windows)

  \author Satofumi KAMIMURA

  $Id: SerialCtrl_win.cpp 910 2009-05-20 09:40:00Z satofumi $
*/

#include "DetectOS.h"
#include <windows.h>

#ifdef MSC
#define snprintf _snprintf
#endif

#undef min
#undef max


class RawSerialCtrl
{
  string error_message_;
  HANDLE hCom_;
  HANDLE hEvent_;
  OVERLAPPED overlapped_;


public:
  RawSerialCtrl::RawSerialCtrl(void)
    : error_message_("no error."),
      hCom_(INVALID_HANDLE_VALUE), hEvent_(INVALID_HANDLE_VALUE)
  {
  }


  const char* RawSerialCtrl::what(void)
  {
    return error_message_.c_str();
  }


  bool RawSerialCtrl::connect(const char* device, long baudrate)
  {
    /* COM ポートを開く */
    enum { NameLength = 11 };
    char adjusted_device[NameLength];
    snprintf(adjusted_device, NameLength, "\\\\.\\%s", device);
    hCom_ = CreateFileA(adjusted_device, GENERIC_READ | GENERIC_WRITE, 0,
                        NULL, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
    if (hCom_ == INVALID_HANDLE_VALUE) {
      error_message_ = string("open failed: ") + device;
      return false;
    }

    hEvent_ = CreateEvent(NULL, TRUE, FALSE, NULL);
    if(hEvent_ == INVALID_HANDLE_VALUE) {
      // !!! エラーメッセージを更新すべき
      return false;
    }
    overlapped_.hEvent = hEvent_;
    overlapped_.Offset = 0;
    overlapped_.OffsetHigh = 0;

    // 通信サイズの更新
    SetupComm(hCom_, 4096 * 8, 4096);

    // ボーレートの変更
    bool ret = setBaudrate(baudrate);
    if (! ret) {
      error_message_ = "fail SerialCtrl::setBaudrate()";
    }
    return true;
  }


  void RawSerialCtrl::disconnect(void)
  {
    if (hCom_ != INVALID_HANDLE_VALUE) {
      CloseHandle(hCom_);
      hCom_ = INVALID_HANDLE_VALUE;
    }
    if (hEvent_) {
      CloseHandle(hEvent_);
      hEvent_ = INVALID_HANDLE_VALUE;
    }
  }


  bool RawSerialCtrl::isConnected(void)
  {
    return (hCom_ == INVALID_HANDLE_VALUE) ? false : true;
  }


  bool RawSerialCtrl::setBaudrate(long baudrate)
  {
    long baudrate_value;
    switch (baudrate) {

    case 4800:
      baudrate_value = CBR_4800;
      break;

    case 9600:
      baudrate_value = CBR_9600;
      break;

    case 19200:
      baudrate_value = CBR_19200;
      break;

    case 38400:
      baudrate_value = CBR_38400;
      break;

    case 57600:
      baudrate_value = CBR_57600;
      break;

    case 115200:
      baudrate_value = CBR_115200;
      break;

    default:
      baudrate_value = baudrate;
    }

    DCB dcb;
    GetCommState(hCom_, &dcb);
    dcb.BaudRate = baudrate_value;
    dcb.ByteSize = 8;
    dcb.Parity = NOPARITY;
    dcb.fParity = FALSE;
    dcb.StopBits = ONESTOPBIT;
    if (SetCommState(hCom_, &dcb) == 0) {
      flush();
      return false;
    } else {
      return true;
    }
  }


  int RawSerialCtrl::send(const char* data, size_t count)
  {
    if (count <= 0) {
      return 0;
    }

    DWORD n;
    if (WriteFile(hCom_, data, (DWORD)count, &n, &overlapped_)) {
      return n;
    } else {
      if (GetLastError() != ERROR_IO_PENDING) {
        return -1;
      }
    }

    WaitForSingleObject(hEvent_, INFINITE);
    GetOverlappedResult(hCom_, &overlapped_, &n, TRUE);

    return n;
  }


  int recv(char data[], int count, int timeout)
  {
    if (count <= 0) {
      return 0;
    }

    DWORD n;
    int filled = 0;
    int read_n;

    do {
      read_n = count - filled;
      if (ReadFile(hCom_, &data[filled], count - filled, &n, &overlapped_)) {
        return filled + n;
      } else {
        if (GetLastError() != ERROR_IO_PENDING) {
          return -1;
        }
      }

      int ret = WaitForSingleObject(hEvent_, timeout);
      if (ret == WAIT_TIMEOUT) {
        break;
      }
      GetOverlappedResult(hCom_, &overlapped_, &n, FALSE);
      filled += n;
    } while ((filled < count) && (n != 0));

    return filled;
  }


  void RawSerialCtrl::flush(void)
  {
    PurgeComm(hCom_,
              PURGE_RXABORT | PURGE_TXABORT | PURGE_RXCLEAR | PURGE_TXCLEAR);
  }
};
