/*!
  \file
  \brief シリアル通信

  \author Satofumi KAMIMURA

  $Id: SerialCtrl.cpp 878 2009-05-13 11:50:25Z satofumi $
*/

#include "SerialCtrl.h"
#include "DetectOS.h"
#include "RingBuffer.h"
#include <string>

using namespace qrk;
using namespace std;


#if defined(WINDOWS_OS)
#include "SerialCtrl_win.cpp"   // Windows (win32) 環境
#else
#include "SerialCtrl_lin.cpp"   // Linux, Mac 環境 (共通)
#endif


struct SerialCtrl::pImpl
{
  string error_message_;
  long baudrate_;
  RawSerialCtrl raw_;
  RingBuffer<char> ring_buffer_; //!< 受信バッファ


  pImpl(void) : error_message_("no error"), baudrate_(0)
  {
  }


  void updateBuffer(void)
  {
    return;
#if 0
    // !!! overlapped 構造体を使っており、読み込めないサイズを指定すると
    // !!! そのデータを読み出すまでデータがブロックするため、コメントアウト

    enum { BufferSize = 1024 };
    char buffer[BufferSize];
    int n = raw_.recv(buffer, BufferSize, 0);
    if (n > 0) {
      ring_buffer_.put(buffer, n);
    }
#endif
  }


  int recv(char* data, size_t count, int timeout)
  {
    if (! isConnected()) {
      error_message_ = "no connection.";
      return -1;
    }
    if (count == 0) {
      return 0;
    }

    size_t filled = 0;

    // バッファにデータがある場合、バッファからデータを格納する
    size_t read_size = std::min(count, ring_buffer_.size());
    ring_buffer_.get(data, read_size);
    filled += read_size;

    // バッファが空の場合、残りのデータはシステムから直接読み込む
    read_size = max(0, static_cast<int>(count - filled));
    if (read_size > 0) {
      int n = raw_.recv(&data[filled], read_size, timeout);
      if (n < 0) {
	error_message_ = raw_.what();
	return n;
      }
      filled += n;
    }
    return static_cast<int>(filled);
  }


  bool isConnected(void)
  {
    return raw_.isConnected();
  }
};


SerialCtrl::SerialCtrl(void) : pimpl(new pImpl)
{
}


SerialCtrl::~SerialCtrl(void)
{
  disconnect();
}


const char* SerialCtrl::what(void)
{
  return pimpl->error_message_.c_str();
}


bool SerialCtrl::connect(const char* device, long baudrate)
{
  disconnect();
  if (! pimpl->raw_.connect(device, baudrate)) {
    pimpl->error_message_ = pimpl->raw_.what();
    return false;
  } else {
    return true;
  }
}


void SerialCtrl::disconnect(void)
{
  return pimpl->raw_.disconnect();
}


bool SerialCtrl::setBaudrate(long baudrate)
{
  if (! pimpl->raw_.setBaudrate(baudrate)) {
    pimpl->error_message_ = pimpl->raw_.what();
    pimpl->baudrate_ = 0;
    return false;
  }
  pimpl->baudrate_ = baudrate;
  return true;
}


long SerialCtrl::baudrate(void)
{
  return pimpl->baudrate_;
}


bool SerialCtrl::isConnected(void)
{
  return pimpl->isConnected();
}


int SerialCtrl::send(const char* data, size_t count)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return 0;
  }

  int n = pimpl->raw_.send(data, static_cast<int>(count));
  if (n < 0) {
    pimpl->error_message_ = pimpl->raw_.what();
  }
  return n;
}


int SerialCtrl::recv(char* data, size_t count, int timeout)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return 0;
  }

  return pimpl->recv(data, count, timeout);
}


size_t SerialCtrl::size(void)
{
  pimpl->updateBuffer();
  return pimpl->ring_buffer_.size();
}


void SerialCtrl::flush(void)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return;
  }

  return pimpl->raw_.flush();
}


void SerialCtrl::clear(void)
{
  pimpl->raw_.flush();
  pimpl->updateBuffer();
  pimpl->ring_buffer_.clear();
}


void SerialCtrl::ungetc(const char ch)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return;
  }

  pimpl->ring_buffer_.ungetc(ch);
}
