/*!
  \file
  \brief 座標系の制御

  \author Satofumi KAMIMURA

  $Id: Coordinate.cpp 772 2009-05-05 06:57:57Z satofumi $

  ツリー構造は、双方向リンクで管理する
*/

#include "Coordinate.h"
#include "Point.h"
#include "MathUtils.h"
#include <map>
#include <list>

using namespace qrk;
using namespace std;


namespace
{
  class RelativePosition
  {
  public:
    Coordinate* coordinate_;    //!< 管理座標系
    Position<long> offset_;      //!< 管理座標系の原点位置
    bool on_module_;            //!< 管理座標系をモジュール上に配置しているか


    RelativePosition(void) : coordinate_(NULL), on_module_(false)
    {
    }


    RelativePosition(Coordinate* parent, const Position<long>& offset,
                     bool on_module = false)
      : coordinate_(parent), offset_(offset), on_module_(on_module)
    {
    }
  };


  const bool OnModule = true;


  typedef map<const Coordinate*, RelativePosition> RelativeMap;
  typedef list<RelativePosition> RelativeList;
}


struct Coordinate::pImpl
{
  Coordinate* own_;
  Coordinate* parent_;
  RelativeMap children_;


  pImpl(Coordinate* own) : own_(own), parent_(NULL)
  {
  }


  void setParent(Coordinate* parent, const qrk::Position<long>& offset)
  {
    parent_ = parent;
    if (parent_) {
      parent_->pimpl->addChild(own_, offset);
    }
  }


  void addChild(Coordinate* child, const qrk::Position<long>& offset,
                bool on_module = false)
  {
    // !!! 自分よりルート側の座標系に接続しないためのループチェックを追加すべき

    child->pimpl->parent_ = own_;
    RelativeMap::iterator p = children_.find(child);
    if (p == children_.end()) {
      // 新規登録
      children_.insert(RelativeMap::value_type
                       (child, RelativePosition(child, offset, on_module)));

    } else {
      // 登録内容を更新
      children_[child] = RelativePosition(child, offset, on_module);
    }
  }


  const vector<Coordinate*> children(void)
  {
    // Coordinate オブジェクトだけの vector を作成して返す
    vector<Coordinate*> children_vector;
    for (RelativeMap::iterator it = children_.begin();
         it != children_.end(); ++it) {
      children_vector.push_back(it->second.coordinate_);
    }
    return children_vector;
  }


  qrk::Position<long> offset(const Coordinate* child) const
  {
    RelativeMap::const_iterator p = children_.find(child);
    if (p == children_.end()) {
      // 指定座標系が登録されてない
      Position<long> dummy;
      return dummy;
    }

    if (! p->second.on_module_) {
      return p->second.offset_;
    } else {
      // モジュール位置を反映させる
      Position<long> module_position = own_->position(parent_);
      Position<long> offset_position = p->second.offset_;
      updatePointPosition(offset_position, module_position);
      return offset_position;
    }
  }


  bool findToRoot(RelativeList& to_coordinate, const Coordinate* coordinate)
  {
    if (own_ == coordinate) {
      // 探索対処が自分自身の場合は、何も登録せずに返す
      return true;
    }

    // ルート側へ探索して、座標系リストを作成する
    const Coordinate* prev_p = own_;
    Coordinate* p = own_;
    for (; p != NULL; prev_p = p, p = p->pimpl->parent_) {
      to_coordinate.push_back(RelativePosition(p, p->offset(prev_p)));
      if (p == coordinate) {
        return true;
      }
    }

    if ((p == NULL) && (coordinate == NULL)) {
      // 時座標系と目標の座標系が NULL のときは、探索は成功
      return true;
    }

    to_coordinate.clear();
    return false;
  }


  void updatePointPosition(Position<long>& position,
                           const Position<long>& offset,
                           bool invert = false) const
  {

    // offset.angle だけ、position の x, y を回転させ、それから、平行移動
    if (! invert) {
      Point<long> move = rotate(position, -offset.to_rad());
      position.x = move.x + offset.x;
      position.y = move.y + offset.y;
      position.angle += offset.angle;

    } else {
      position.angle -= offset.angle;
      Point<long> move = rotate(offset, position.to_rad());
      position += Position<long>(move.x, move.y, deg(0));
    }
  }


  Point<long> rotate(const Position<long>& point, double radian) const
  {
    int x = static_cast<long>
      (ceil((point.x * cos(-radian)) - (point.y * sin(-radian))));
    int y = static_cast<long>
      (ceil((point.x * sin(-radian)) + (point.y * cos(-radian))));

    return Point<long>(x, y);
  }
};


Coordinate::Coordinate(void) : pimpl(new pImpl(this))
{
}


Coordinate::Coordinate(Coordinate* parent, const qrk::Position<long>& offset)
  : pimpl(new pImpl(this))
{
  pimpl->setParent(parent, offset);
}


Coordinate::~Coordinate(void)
{
}


void Coordinate::updatePointPosition(Position<long>& position,
                                     const Position<long>& offset,
                                     bool invert) const
{
  pimpl->updatePointPosition(position, offset, invert);
}


Coordinate* Coordinate::parent(void)
{
  return pimpl->parent_;
}


const vector<Coordinate*> Coordinate::children(void)
{
  return pimpl->children();
}


void Coordinate::addChild(Coordinate* child)
{
  pimpl->addChild(child, Position<long>(0, 0, deg(0)));
}


void Coordinate::addChild(Coordinate* child, const qrk::Position<long>& offset)
{
  pimpl->addChild(child, offset);
}


void Coordinate::addChildToModule(Coordinate* child)
{
  addChildToModule(child, Position<long>(0, 0, deg(0)));
}


void Coordinate::addChildToModule(Coordinate* child,
                                  const qrk::Position<long>& offset)
{
  pimpl->addChild(child, offset, OnModule);
}


qrk::Position<long> Coordinate::offset(const Coordinate* child) const
{
  return pimpl->offset(child);
}


qrk::Position<long> Coordinate::originPosition(Coordinate* coordinate)
{
  return pointPosition(coordinate, qrk::Position<long>(0, 0, deg(0)));
}


qrk::Position<long> Coordinate::pointPosition(Coordinate* coordinate,
                                             const qrk::Position<long>& position)
{
  // 指定座標系までの座標系リストを作成する
  RelativeList to_target;
  if (pimpl->findToRoot(to_target, coordinate)) {

    Position<long> converted_position = position;
    for (RelativeList::iterator it = to_target.begin();
         it != to_target.end(); ++it) {
      pimpl->updatePointPosition(converted_position, it->offset_);
    }
    return converted_position;
  }

  if (coordinate->pimpl->findToRoot(to_target, this)) {

    Position<long> converted_position = position;
    for (RelativeList::reverse_iterator it = to_target.rbegin();
         it != to_target.rend(); ++it) {
      pimpl->updatePointPosition(converted_position, it->offset_, true);
    }
    return converted_position;
  }

  // 見付からなければ、例外を投げる
  // !!! 今は、まだ投げていない
  Position<long> dummy;
  return dummy;
}
