/*!
  \file
  \brief シリアル通信 (Windows 実装)

  Serial Communication Interface 制御


  \author Satofumi KAMIMURA

  $Id: serial_ctrl_win.c 882 2009-05-13 19:30:41Z satofumi $

  \todo 直前のタイムアウト設定と同じならば、再設定しないようにする
*/

#include "serial_ctrl.h"
#include "serial_errno.h"
#include "delay.h"
#include <stdio.h>


/* 接続 */
int serial_connect(serial_t *serial, const char *device, long baudrate)
{
  char adjusted_device[16];

  /* COM ポートを開く */
  _snprintf(adjusted_device, 16, "\\\\.\\%s", device);
  serial->hCom_ = CreateFileA(adjusted_device, GENERIC_READ | GENERIC_WRITE, 0,
			      NULL, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
  if (serial->hCom_ == INVALID_HANDLE_VALUE) {
    fprintf(stderr, "open failed: %s\n", device);
    return -1;
  }

  serial->hEvent_ = CreateEvent(NULL, TRUE, FALSE, NULL);
  if(! serial->hEvent_) {
    // !!! エラーメッセージを更新すべき
    return -1;
  }
  serial->overlapped_.hEvent = serial->hEvent_;
  serial->overlapped_.Offset = 0;
  serial->overlapped_.OffsetHigh = 0;

  /* 通信サイズの更新 */
  SetupComm(serial->hCom_, 4096 * 4, 4096);

  /* ボーレートの変更 */
  serial_setBaudrate(serial, baudrate);

  /* シリアル制御構造体の初期化 */
  serial->last_ch_ = '\0';
  serial->current_timeout_ = 0;

  return 0;
}


/* 切断 */
void serial_disconnect(serial_t *serial)
{
  if (serial->hCom_) {
    CloseHandle(serial->hCom_);
    serial->hCom_ = INVALID_HANDLE_VALUE;
  }
  if (serial->hEvent_) {
    CloseHandle(serial->hEvent_);
    serial->hEvent_ = INVALID_HANDLE_VALUE;
  }
}


int serial_isConnected(serial_t *serial)
{
  return (serial->hCom_ == INVALID_HANDLE_VALUE) ? 0 : 1;
}


/* ボーレートの変更 */
int serial_setBaudrate(serial_t *serial, long baudrate)
{
  long baudrate_value;

  switch (baudrate) {

  case 4800:
    baudrate_value = CBR_4800;
    break;

  case 9600:
    baudrate_value = CBR_9600;
    break;

  case 19200:
    baudrate_value = CBR_19200;
    break;

  case 38400:
    baudrate_value = CBR_38400;
    break;

  case 57600:
    baudrate_value = CBR_57600;
    break;

  case 115200:
    baudrate_value = CBR_115200;
    break;

  default:
    baudrate_value = baudrate;
  }

  DCB dcb;
  GetCommState(serial->hCom_, &dcb);
  dcb.BaudRate = baudrate_value;
  dcb.ByteSize = 8;
  dcb.Parity = NOPARITY;
  dcb.fParity = FALSE;
  dcb.StopBits = ONESTOPBIT;
  SetCommState(serial->hCom_, &dcb);

  serial_clear(serial);

  return 0;
}


/* 送信 */
int serial_send(serial_t *serial, const char *data, int data_size)
{
  DWORD n;

  if (! serial_isConnected(serial)) {
    return SerialConnectionFail;
  }

  if (WriteFile(serial->hCom_, data, (DWORD)data_size,
		&n, &serial->overlapped_)) {
    return n;
  } else {
    if (GetLastError() != ERROR_IO_PENDING) {
      return -1;
    }
  }
  
  WaitForSingleObject(serial->hEvent_, INFINITE);
  GetOverlappedResult(serial->hCom_, &serial->overlapped_, &n, TRUE);

  return n;
}


/* 受信 */
int serial_recv(serial_t *serial, char* data, int data_size_max, int timeout)
{
  DWORD n;
  int filled = 0;

  if (data_size_max <= 0) {
    return 0;
  }

  /* 書き戻した１文字があれば、書き出す */
  filled = 0;
  if (serial->last_ch_ != '\0') {
    data[0] = serial->last_ch_;
    serial->last_ch_ = '\0';
    ++filled;
    --data_size_max;
  }

  // 要求サイズだけ読み出す
  if (ReadFile(serial->hCom_, &data[filled], (DWORD)data_size_max,
	       &n, &serial->overlapped_)) {
    return (n + filled);
  } else {
    if (GetLastError() != ERROR_IO_PENDING) {
      return -1;
    }
  }
  // 読み出しのタイムアウト設定
  WaitForSingleObject(serial->hEvent_, timeout);
  GetOverlappedResult(serial->hCom_, &serial->overlapped_, &n, FALSE);

  return (n + filled);
}


/* １文字書き戻す */
void serial_ungetc(serial_t *serial, char ch)
{
  serial->last_ch_ = ch;
}


void serial_clear(serial_t* serial)
{
  PurgeComm(serial->hCom_,
	    PURGE_RXABORT | PURGE_TXABORT | PURGE_RXCLEAR | PURGE_TXCLEAR);
  serial->last_ch_ = '\0';
}
