/*!
  \file
  \brief URG Viewer Sample

  \author Satofumi KAMIMURA

  $Id: UrgViewerWindow.cpp 292 2008-10-23 14:33:10Z satofumi $

  \todo Qt 版の system ライブラリが sdl 版と互換になるように調整する
  \todo space で再生の pause/resume ができるようにする
  \todo 再生を最初から行えるようにする
*/

#include "UrgViewerWindow.h"
#include "UrgDrawWidget.h"
#include "ConnectThread.h"
#include "SerialConnectWidget.h"
#include "mUrgCtrl.h"
#include "getTicks.h"
#include "MonitorModeManager.h"
#include <QTimer>
#include <QShortcut>
#include <QMessageBox>
#include <QSettings>
#include <QCloseEvent>

using namespace qrk;


namespace
{
  const char* Organization = "Hokuyo LTD.";
  const char* Application = "URG Viewer";
};


struct UrgViewerWindow::pImpl
{
  UrgViewerWindow* parent_;

  mUrgCtrl urg_;

  UrgDrawWidget urg_draw_widget_;
  SerialConnectWidget connect_widget_;
  ConnectThread connect_thread_;

  QTimer capture_timer_;


  pImpl(int argc, char *argv[], UrgViewerWindow* parent)
    : parent_(parent), urg_(argc, argv), connect_widget_(&urg_),
      connect_thread_(&connect_widget_)
  {
  }


  void initializeForm(void)
  {
    // ウィジットの配置
    parent_->dummy_label_1_->hide();
    parent_->dummy_label_2_->hide();
    parent_->main_layout_->addWidget(&urg_draw_widget_);
    parent_->connect_layout_->addWidget(&connect_widget_);

    // メニュー項目
    connect(parent_->action_quit_, SIGNAL(triggered()),
            parent_, SLOT(close()));
    connect(parent_->action_about_, SIGNAL(triggered()),
            parent_, SLOT(aboutApplication()));

    // URG デバイス
    connect(&connect_widget_, SIGNAL(connectEvent()),
            parent_, SLOT(connectDevice()));
    connect(&connect_widget_, SIGNAL(disconnectEvent()),
            parent_, SLOT(disconnectDevice()));
    connect(&capture_timer_, SIGNAL(timeout()),
            parent_, SLOT(captureHandler()));

    // ステータスバー表示
    connect(&urg_draw_widget_, SIGNAL(position(bool, long, long)),
            parent_, SLOT(updateStatusBar(bool, long, long)));

    // キー割り付け
    (void) new QShortcut(Qt::Key_Return, parent_, SLOT(initializeView()));
    (void) new QShortcut(Qt::Key_Less, parent_, SLOT(zoomSmaller()));
    (void) new QShortcut(Qt::Key_Comma, parent_, SLOT(zoomSmaller()));
    (void) new QShortcut(Qt::Key_Greater, parent_, SLOT(zoomLarger()));
    (void) new QShortcut(Qt::Key_Period, parent_, SLOT(zoomLarger()));
  }


  void saveSettings(void)
  {
    QSettings settings(Organization, Application);

    settings.setValue("geometry", parent_->saveGeometry());
  }


  void loadSettings(void)
  {
    QSettings settings(Organization, Application);

    parent_->restoreGeometry(settings.value("geometry").toByteArray());
  }


  void connectDevice(void)
  {
    // URG 設定
    urg_.setTimestamp(getTicks());
    urg_.setCaptureMode(AutoCapture);
    int scan_msec = urg_.scanMsec();

    // データの取得開始
    capture_timer_.start(scan_msec);
  }


  void disconnectDevice(void)
  {
    // データの取得停止
    capture_timer_.stop();
    urg_draw_widget_.clear();
  }


  void captureHandler(void)
  {
    urg_draw_widget_.setUrgData(&urg_);
    //int timestamp = urg_.lastTimsatamp();

    // 一定間隔での描画
    // !!! FPS を定義する
    // !!! 可能ならば、URG のタイムスタンプを利用する
    urg_draw_widget_.redraw();
  }


  void setPlayConnection(void)
  {
    // 接続操作を disable にする
    connect_widget_.setEnabled(false);

    // 接続処理
    QTimer::singleShot(1000, parent_, SLOT(autoPlay()));
  }
};


UrgViewerWindow::UrgViewerWindow(int argc, char *argv[])
  : pimpl(new pImpl(argc, argv, this))
{
  setupUi(this);
  pimpl->initializeForm();
  pimpl->loadSettings();

  if (MonitorModeManager::singleton()->mode() == MonitorModeManager::Play) {
    // 接続操作を disable にして、接続を行っておく
    pimpl->setPlayConnection();
  }
}


UrgViewerWindow::~UrgViewerWindow(void)
{
  pimpl->saveSettings();
}


void UrgViewerWindow::connectDevice(void)
{
  pimpl->connectDevice();
}


void UrgViewerWindow::disconnectDevice(void)
{
  pimpl->disconnectDevice();
}


void UrgViewerWindow::captureHandler(void)
{
  pimpl->captureHandler();
}


void UrgViewerWindow::initializeView(void)
{
  pimpl->urg_draw_widget_.initializeView();
}


void UrgViewerWindow::updateStatusBar(bool active, long x_mm, long y_mm)
{
  if (! active) {
    statusBar()->clearMessage();
    return;
  }

  QString message = QString("X: %1 [mm], Y: %2 [mm]").arg(x_mm).arg(y_mm);
  statusBar()->showMessage(message, 60 * 1000);
}


void UrgViewerWindow::aboutApplication(void)
{
  QMessageBox::about(this, tr("About URG Viewer"),
                     tr("<h2>URG Viewer ($Rev: 292 $)</h2>"
                        "<p>Demo Application for URG sensor</p>"
                        "<p>Report bugs to "
                        "&lt;s-kamimura@hokuyo-aut.co.jp&gt;</p>"));
}


void UrgViewerWindow::autoPlay(void)
{
  pimpl->connect_thread_.start();
}


void UrgViewerWindow::zoomSmaller(void)
{
  pimpl->urg_draw_widget_.updateZoomRatio(+1);
}


void UrgViewerWindow::zoomLarger(void)
{
  pimpl->urg_draw_widget_.updateZoomRatio(-1);
}


void UrgViewerWindow::closeEvent(QCloseEvent* event)
{
  pimpl->connect_thread_.terminate();
}
