/*!
  \file
  \brief URG データの描画ウィジット

  \author Satofumi KAMIMURA

  $Id: UrgDrawWidget.cpp 298 2008-10-26 07:37:06Z satofumi $

  \todo 接続できないときに、ボタンが戻るように追加する
  \todo 強度データ用の表示切替えコンボボックスを付加
  \todo ポリゴン形式での表示方法を実装する
  \todo グリッドに数値と単位系を描画。画面左端と、画面下
  \todo 描画する色を変更できるようにする
*/

#include "UrgDrawWidget.h"
#include "RangeSensor.h"
#include "convert2D.h"
//#include "GridUnit.h"
#include "MathUtils.h"
#include <QMouseEvent>
#include <deque>

using namespace qrk;


namespace
{
  const double DefaultPixelPerMm = 20.0;

  struct DrawData
  {
    std::vector<Grid<long> > point_data;
    int timestamp;


    DrawData(std::vector<Grid<long> >& point_data_,
             int timestamp_)
      : timestamp(timestamp_)
    {
      std::swap(point_data, point_data_);
    }
  };
};


struct UrgDrawWidget::pImpl
{
  enum {
    MinimumWidth = 100,
    MinimumHeight = 100,

    GridLength = 30 * 1000,     // [mm]
  };

  UrgDrawWidget* parent_;

  Grid<double> view_center_;
  double pixel_per_mm_;
  DrawMode draw_mode_;
  size_t draw_period_;

  typedef std::deque<DrawData> DataArray;
  DataArray draw_data_;
  int last_timestamp_;
  int current_timestamp_;
  size_t width_;
  size_t height_;

  QColor clear_color_;
  Position<long> rotate_offset_;
  QPoint clicked_position_;
  bool now_pressed_;

  //GridUnit grid_unit_;


  pImpl(UrgDrawWidget* parent)
    : parent_(parent),
      pixel_per_mm_(DefaultPixelPerMm), draw_mode_(Lines), draw_period_(0),
      last_timestamp_(0), current_timestamp_(0),
      width_(MinimumWidth), height_(MinimumHeight),
      clear_color_(Qt::white), rotate_offset_(Position<long>(0, 0, deg(90))),
      now_pressed_(false)
      //grid_unit_("segment.ppm", "0123456789mc[]")
  {
  }


  void initializeForm(void)
  {
    parent_->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    parent_->setMinimumSize(MinimumWidth, MinimumHeight);

    parent_->setMouseTracking(true);
  }


  void removeOldData(int timestamp)
  {
    int index = 0;
    for (DataArray::iterator it = draw_data_.begin();
         it != draw_data_.end(); ++it, ++index) {
      if (it->timestamp > (timestamp - static_cast<int>(draw_period_))) {
        break;
      }
    }

    if (index > 0) {
      DataArray::iterator it = draw_data_.begin();
      draw_data_.erase(it, it + index);
    }
  }


  void initializeGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glEnable(GL_CULL_FACE);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);
    glDisable(GL_DEPTH_TEST);

    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }


  void resizeGL(int width, int height)
  {
    glViewport(0, 0, width, height);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    double aspect = 1.0 * width / height;
    glOrtho(-1.0 * aspect, +1.0 * aspect, -1.0, +1.0,
            std::numeric_limits<int>::min(), std::numeric_limits<int>::max());

    glMatrixMode(GL_MODELVIEW);
    width_ = width;
    height_ = height;
  }


  void paintGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glLoadIdentity();

    // 軸の描画
    drawAxis();

    // データの描画
    drawData();
  }


  void drawAxis(void)
  {
    double zoom_ratio = zoomRatio();
    Grid<double> offset = drawOffset(zoom_ratio);

    // 補助線の描画
    drawSubAxis(offset);

    glColor3d(0.3, 0.3, 0.3);
    glBegin(GL_LINES);

    // Y 軸
    glVertex2d(-offset.x, -1.0);
    glVertex2d(-offset.x, +1.0);

    // X 軸
    double aspect = 1.0 * width_ / height_;
    glVertex2d(-aspect, -offset.y);
    glVertex2d(+aspect, -offset.y);

    // 背後 45 [deg] の補助線
    glVertex2d(-offset.x, -offset.y);
    glVertex2d((-GridLength * zoom_ratio) - offset.x,
               (-GridLength * zoom_ratio) - offset.y);
    glVertex2d(-offset.x, -offset.y);
    glVertex2d((+GridLength * zoom_ratio) - offset.x,
               (-GridLength * zoom_ratio) - offset.y);
    glEnd();

    // 円の補助線を描画
    for (int i = 0; i <= GridLength; i += 1000) {
      drawSubCircle(i, offset);
    }

    // 単位の表示
    // !!!
    //grid_unit_.drawUnit(0.0, 0.0, 987654, GridUnit::Unit_mm);
  }


  void drawSubCircle(int radius, const Grid<double>& offset)
  {
    double pixel_radius = (radius / pixel_per_mm_) / (height_ / 2.0);

    glBegin(GL_LINE_STRIP);
    for (int theta = -45; theta <= (180 + 45); theta += 2) {

      double radian = M_PI * theta / 180.0;
      double x = pixel_radius * cos(radian);
      double y = pixel_radius * sin(radian);
      glVertex2d(x - offset.x, y - offset.y);
    }
    glEnd();
  }


  void drawSubAxis(const Grid<double>& offset)
  {
    Grid<int> center(width_ / 2, height_ / 2);
    Grid<long> first(static_cast<int>((-center.x * pixel_per_mm_)
                                      - view_center_.x),
                     static_cast<int>((-center.y * pixel_per_mm_)
                                      - view_center_.y));
    Grid<long> last(static_cast<int>((+center.x * pixel_per_mm_)
                                      - view_center_.x),
                     static_cast<int>((+center.y * pixel_per_mm_)
                                      - view_center_.y));

    const double threshold[] = { 4.0, 16.0, 32.0 };
    const int interval[] = { 10, 100, 1000 };
    const double color[] = { 0.8, 0.6, 0.3 };
    size_t n = sizeof(threshold) / sizeof(threshold[0]);
    for (size_t i = 0; i < n; ++i) {
      const double draw_threshold = threshold[i];
      if (pixel_per_mm_ > draw_threshold) {
        continue;
      }
      double alpha = 1.0 - (pixel_per_mm_ / draw_threshold);
      glColor4d(color[i], color[i], color[i], alpha);
      drawSubGridLine(interval[i], offset, first, last);
    }
  }


  void drawSubGridLine(int interval, const Grid<double>& offset,
                       const Grid<long>& first, const Grid<long>& last)
  {
    glBegin(GL_LINES);
    for (int x = (first.x / interval) - 1; x < ((last.x / interval) + 1); ++x) {
      double draw_x = (interval * x / pixel_per_mm_) / (height_ / 2.0);
      glVertex2d(draw_x - offset.x, -1.0);
      glVertex2d(draw_x - offset.x, +1.0);
    }

    double aspect = 1.0 * width_ / height_;
    for (int y = (first.y / interval) - 1; y < ((last.y / interval) + 1); ++y) {
      double draw_y = (interval * y / pixel_per_mm_) / (height_ / 2.0);
      glVertex2d(-aspect, draw_y - offset.y);
      glVertex2d(+aspect, draw_y - offset.y);
    }
    glEnd();
  }


  void drawData(void)
  {
    // !!! 時間が経過するほど、薄い色で表示するようにする

    double zoom_ratio = zoomRatio();
    Grid<double> offset = drawOffset(zoom_ratio);


    if (draw_mode_ == Lines) {
      // 中心から測定点への直線を描画
      glColor3d(0.0, 0.0, 1.0);
      glPointSize(1.0);

      glBegin(GL_LINES);
      for (DataArray::iterator line_it = draw_data_.begin();
           line_it != draw_data_.end(); ++line_it) {

        std::vector<Grid<long> >& line_data = draw_data_.front().point_data;;
        std::vector<Grid<long> >::iterator end_it = line_data.end();
        for (std::vector<Grid<long> >::iterator it = line_data.begin();
             it != end_it; ++it) {

          double x = it->x * zoom_ratio;
          double y = it->y * zoom_ratio;
          glVertex2d(-offset.x, -offset.y);
          glVertex2d(x - offset.x, y - offset.y);
        }
      }
      glEnd();

    } else if (draw_mode_ == Polygon) {
      // !!!
      // !!! これを実現するには、convert2D 時に捨ててしまった情報が必要
      // !!! 実装方法を見直すべき
      // !!!
    }

    // 測定点の描画
    glColor3d(1.0, 0.0, 0.0);
    double mm_pixel = 1.0 / pixel_per_mm_;
    if (mm_pixel < 1.0) {
      mm_pixel = 1.0;
    }
    glPointSize(mm_pixel);

    glBegin(GL_POINTS);
    for (DataArray::iterator line_it = draw_data_.begin();
         line_it != draw_data_.end(); ++line_it) {

      std::vector<Grid<long> >::iterator end_it = line_it->point_data.end();
      for (std::vector<Grid<long> >::iterator it = line_it->point_data.begin();
           it != end_it; ++it) {

        double x = it->x * zoom_ratio;
        double y = it->y * zoom_ratio;
        glVertex2d(x - offset.x, y - offset.y);
      }
    }
    glEnd();
  }


  // [mm] -> [pixel]
  double zoomRatio(void)
  {
    return (1.0 / pixel_per_mm_ / (height_ / 2.0));
  }


  Grid<double> drawOffset(double zoom_ratio)
  {
    return Grid<double>(-view_center_.x * zoom_ratio,
                        -view_center_.y * zoom_ratio);
  }


  void setClickedPosition(QMouseEvent* event)
  {
    clicked_position_ = event->pos();
  }


  void updateZoomRatio(int steps)
  {
    double zoom = parent_->zoomRatio();
    zoom *= pow(1.1, steps);

    if (zoom > 500.0) {
      zoom = 500.0;
    } else if (zoom < 0.05) {
      zoom = 0.05;
    }
    parent_->setZoomRatio(zoom);
  }
};


UrgDrawWidget::UrgDrawWidget(QWidget* parent)
  : QGLWidget(parent), pimpl(new pImpl(this))
{
  pimpl->initializeForm();
}


UrgDrawWidget::~UrgDrawWidget(void)
{
}


void UrgDrawWidget::clear(void)
{
  pimpl->draw_data_.clear();
  redraw();
}


void UrgDrawWidget::initializeView(void)
{
  pimpl->pixel_per_mm_ = DefaultPixelPerMm;
  pimpl->view_center_ = Grid<double>(0.0, 0.0);
  redraw();
}


void UrgDrawWidget::redraw(int timestamp)
{
  pimpl->current_timestamp_ =
    (timestamp == NoneSpecified) ? pimpl->last_timestamp_ : timestamp;
  updateGL();
}


void UrgDrawWidget::setUrgData(qrk::RangeSensor* sensor)
{
  std::vector<long> data;
  int timestamp = 0;
  int n = sensor->capture(data, &timestamp);
  if (n <= 0) {
    return;
  }

  std::vector<Grid<long> > point_data;
  qrk::convert2D(point_data, sensor, data, pimpl->rotate_offset_);

  // !!! 以降の処理は、繰り返されているので、まとめるべき
  if (! point_data.empty()) {
    pimpl->removeOldData(timestamp);
    pimpl->draw_data_.push_back(DrawData(point_data, timestamp));
    pimpl->last_timestamp_ = timestamp;
  }
}


void UrgDrawWidget::setUrgData(std::vector<long>& data,
                               const RangeSensor* sensor, int timestamp)
{
  std::vector<Grid<long> > point_data;
  qrk::convert2D(point_data, sensor, data, pimpl->rotate_offset_);

  if (! point_data.empty()) {
    pimpl->removeOldData(timestamp);
    pimpl->draw_data_.push_back(DrawData(point_data, timestamp));
    pimpl->last_timestamp_ = timestamp;
  }
}


void UrgDrawWidget::setUrgData(std::vector<Grid<long> >& data, int timestamp)
{
  if (! data.empty()) {
    pimpl->removeOldData(timestamp);
    pimpl->draw_data_.push_back(DrawData(data, timestamp));
    pimpl->last_timestamp_ = timestamp;
  }
}


void UrgDrawWidget::setDrawMode(DrawMode mode)
{
  pimpl->draw_mode_ = mode;
}


void UrgDrawWidget::setDrawPeriod(size_t msec)
{
  pimpl->draw_period_ = msec;
}


void UrgDrawWidget::setZoomRatio(double pixel_per_mm)
{
  pimpl->pixel_per_mm_ = pixel_per_mm;
  redraw();
}


double UrgDrawWidget::zoomRatio(void)
{
  return pimpl->pixel_per_mm_;
}


void UrgDrawWidget::updateZoomRatio(int steps)
{
  pimpl->updateZoomRatio(steps);
}


void UrgDrawWidget::setViewCenter(const Grid<long>& point)
{
  pimpl->view_center_.x = point.x;
  pimpl->view_center_.y = point.y;
}


Grid<long> UrgDrawWidget::viewCenter(void)
{
  return Grid<long>(static_cast<long>(pimpl->view_center_.x),
                    static_cast<long>(pimpl->view_center_.y));
}


void UrgDrawWidget::initializeGL(void)
{
  pimpl->initializeGL();
}


void UrgDrawWidget::resizeGL(int width, int height)
{
  pimpl->resizeGL(width, height);
}


void UrgDrawWidget::paintGL(void)
{
  pimpl->paintGL();
}


void UrgDrawWidget::mousePressEvent(QMouseEvent* event)
{
  pimpl->now_pressed_ = true;
  pimpl->setClickedPosition(event);
}


void UrgDrawWidget::mouseMoveEvent(QMouseEvent* event)
{
  int x = event->x();
  int y = event->y();

  if (pimpl->now_pressed_) {
    int dx = x - pimpl->clicked_position_.x();
    int dy = y - pimpl->clicked_position_.y();

    pimpl->view_center_.x += dx * pimpl->pixel_per_mm_;
    pimpl->view_center_.y -= dy * pimpl->pixel_per_mm_;

    pimpl->setClickedPosition(event);
  }

  // カーソル位置の座標をシグナルで送信する
  if ((x < 0) || (x >= static_cast<int>(pimpl->width_)) ||
      (y < 0) || (y >= static_cast<int>(pimpl->height_))) {
    emit position(false, -1, -1);
    return;
  }

  int center_x = pimpl->width_ / 2;
  int center_y = pimpl->height_ / 2;
  int x_mm = static_cast<int>(((x - center_x) * pimpl->pixel_per_mm_)
                              - pimpl->view_center_.x);
  int y_mm = static_cast<int>((-(y - center_y) * pimpl->pixel_per_mm_)
                              - pimpl->view_center_.y);
  emit position(true, x_mm, y_mm);
}


void UrgDrawWidget::mouseReleaseEvent(QMouseEvent* event)
{
  static_cast<void>(event);

  pimpl->now_pressed_ = false;
  redraw();
}


void UrgDrawWidget::wheelEvent(QWheelEvent* event) {

  int degrees = event->delta() / 8;
  int steps = degrees / 15;

  event->accept();
  updateZoomRatio(steps);
}
