/*!
  \file
  \brief Wii 状態の表示ウィジット

  \author Satofumi KAMIMURA

  $Id$

  \todo 接続中は、アニメーションを表示する
  \todo 加速度の表示フォーマットを変更する。小数点以下２桁まで、符号そろえ
*/

#include <QTimer>
#include <QShortcut>
#include "WiiStateWidget.h"
#include "WiiJoystick.h"

using namespace qrk;


namespace {

  class ButtonLabel {
  public:
    QLabel* label_;
    int index_;

    ButtonLabel(QLabel* label, int index) : label_(label), index_(index) {
    }
  };

  typedef std::vector<ButtonLabel> ButtonLabels;
};

struct WiiStateWidget::pImpl {

  WiiJoystick wii_;
  ButtonLabels button_labels_;
  QTimer* timer_;

  pImpl(void) : timer_(NULL) {
  }

  void initializeForm(WiiStateWidget* parent) {

    // ラベル情報の初期化
    button_labels_.push_back(ButtonLabel(parent->a_value_label_,
                                         WiiJoystick::BUTTON_A));
    button_labels_.push_back(ButtonLabel(parent->b_value_label_,
                                         WiiJoystick::BUTTON_B));
    button_labels_.push_back(ButtonLabel(parent->one_value_label_,
                                         WiiJoystick::BUTTON_1));
    button_labels_.push_back(ButtonLabel(parent->two_value_label_,
                                         WiiJoystick::BUTTON_2));
    button_labels_.push_back(ButtonLabel(parent->plus_value_label_,
                                         WiiJoystick::BUTTON_PLUS));
    button_labels_.push_back(ButtonLabel(parent->minus_value_label_,
                                         WiiJoystick::BUTTON_MINUS));
    button_labels_.push_back(ButtonLabel(parent->home_value_label_,
                                         WiiJoystick::BUTTON_HOME));

    // 接続イベント
    connect(parent->connect_button_, SIGNAL(clicked(bool)),
            parent, SLOT(connectionHandler(bool)));

    // LED イベントの接続
    connect(parent->led3_checkbox_, SIGNAL(clicked()),
            parent, SLOT(ledHandler()));
    connect(parent->led2_checkbox_, SIGNAL(clicked()),
            parent, SLOT(ledHandler()));
    connect(parent->led1_checkbox_, SIGNAL(clicked()),
            parent, SLOT(ledHandler()));
    connect(parent->led0_checkbox_, SIGNAL(clicked()),
            parent, SLOT(ledHandler()));

    // Rumble イベントの接続
    connect(parent->rumble_checkbox_, SIGNAL(clicked()),
            parent, SLOT(rumbleHandler()));

    // ボタンで終了
    connect(parent->close_button_, SIGNAL(clicked()), parent, SLOT(close()));

    // Ctrl-q で終了
    (void) new QShortcut(Qt::CTRL + Qt::Key_Q, parent, SLOT(close()));
  }


  // 操作の enable / disable
  void setEnabled(WiiStateWidget* parent, bool enable) {

    parent->buttons_group_->setEnabled(enable);
    parent->acceleration_group_->setEnabled(enable);
    parent->led_group_->setEnabled(enable);
    parent->rumble_group_->setEnabled(enable);
  }


  // 加速度の浮動小数点表示
  void setAccelerationValue(QLabel* label, double value) {

    enum { BufferSize = 6 };
    char buffer[BufferSize] = "-0.00";
    snprintf(buffer, BufferSize, "% .2f", value);
    label->setText(QString(buffer));
  }
};


WiiStateWidget::WiiStateWidget(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl) {
  setupUi(this);

  // フォームの初期化
  pimpl->initializeForm(this);

  // ポーリング用のタイマーを起動
  pimpl->timer_ = new QTimer(this);
  connect(pimpl->timer_, SIGNAL(timeout()), this, SLOT(statusUpdate()));
}


WiiStateWidget::~WiiStateWidget(void) {
}


// 接続
void WiiStateWidget::connectionHandler(bool checked) {

  if (checked) {
    // 接続
    if (! pimpl->wii_.connect()) {

      // エラーメッセージの表示
      qWarning("WiiJoystick::connect: %s\n", pimpl->wii_.what());
      connect_button_->setChecked(false);
      return;
    }

    enum {
      AccelerationAverageSize = 8,
    };
    pimpl->wii_.setAccelerationAverageSize(AccelerationAverageSize);

    // 操作を許可
    pimpl->setEnabled(this, true);
    pimpl->timer_->start(100);

    // ラベル表示を変更
    connect_button_->setText(tr("Disconnect"));

  } else {
    // 切断
    pimpl->timer_->stop();
    pimpl->wii_.disconnect();

    // 操作を不許可
    pimpl->setEnabled(this, false);

    // ラベル表示を変更
    connect_button_->setText(tr("Connect"));
  }
}


// LED
void WiiStateWidget::ledHandler(void) {

  unsigned char led_value =
    (led3_checkbox_->isChecked() ? 0x08 : 0x00) |
    (led2_checkbox_->isChecked() ? 0x04 : 0x00) |
    (led1_checkbox_->isChecked() ? 0x02 : 0x00) |
    (led0_checkbox_->isChecked() ? 0x01 : 0x00);

  pimpl->wii_.setLed(led_value);
}


// 振動
void WiiStateWidget::rumbleHandler(void) {

  pimpl->wii_.setRumble(rumble_checkbox_->isChecked());
}


// 状態の更新
void WiiStateWidget::statusUpdate(void) {

  if (! pimpl->wii_.isConnected()) {
    // 接続されてないので、戻る
    return;
  }

  // 全情報を取得し、テキストを更新する
  int y_axis = pimpl->wii_.getAxisValue(WiiJoystick::AxisY);
  up_value_label_->setText((y_axis > 0) ? "On" : "---");
  down_value_label_->setText((y_axis < 0) ? "On" : "---");

  int x_axis = pimpl->wii_.getAxisValue(WiiJoystick::AxisX);
  left_value_label_->setText((x_axis < 0) ? "On" : "---");
  right_value_label_->setText((x_axis > 0) ? "On" : "---");

  // ボタン情報の更新
  for (ButtonLabels::iterator it = pimpl->button_labels_.begin();
       it != pimpl->button_labels_.end(); ++it) {
    bool pressed = pimpl->wii_.isButtonPressed(it->index_);
    it->label_->setText(pressed ? "On" : "---");
  }

  // 加速度情報の更新
  Grid3D<double> acceleration;
  pimpl->wii_.getAcceleration(acceleration);
  pimpl->setAccelerationValue(x_value_label_, acceleration.x);
  pimpl->setAccelerationValue(y_value_label_, acceleration.y);
  pimpl->setAccelerationValue(z_value_label_, acceleration.z);
}
