/*!
  \file
  \brief Wii の位置情報を表示

  \author Satofumi KAMIMURA

  $Id: WiiPoseWidget.cpp 130 2008-07-25 06:00:54Z satofumi $

  \todo Wii の加速度の値を表示する
  \todo Wii が向いている角度を表示する
*/

#include <cmath>
#include <QTimer>
#include <QShortcut>
#include "WiiPoseWidget.h"
#include "WiiJoystick.h"
#include "WiiPoseDraw.h"

using namespace qrk;


struct WiiPoseWidget::pImpl {
  QTimer* redraw_timer_;
  WiiJoystick wii_;
  WiiPoseDraw* paint_widget_;

  pImpl(WiiPoseWidget* parent) : redraw_timer_(new QTimer(parent)) {
  }

  void initializeForm(WiiPoseWidget* parent) {

    // 接続ボタンの処理
    connect(parent->connect_button_, SIGNAL(clicked(bool)),
            parent, SLOT(handleConnect(bool)));

    // Wii 情報の更新用
    connect(redraw_timer_, SIGNAL(timeout()), parent, SLOT(updatePose()));
  }

  void clearForm(WiiPoseWidget* parent) {
    parent->battery_label_->setText(tr("battery:  --[%]"));
  }
};


WiiPoseWidget::WiiPoseWidget(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this)) {
  setupUi(this);

  pimpl->paint_widget_ = new WiiPoseDraw(widget_);

  pimpl->initializeForm(this);
  pimpl->clearForm(this);

  // Ctrl-q で終了できるようにする
  (void) new QShortcut(Qt::CTRL + Qt::Key_Q, this, SLOT(close()));
}


WiiPoseWidget::~WiiPoseWidget(void) {
}



void WiiPoseWidget::handleConnect(bool checked) {

  if (checked) {
#if 0
    // !!! 動かない。エラーが発生して戻る
    // !!! 使いかたを間違えたかな？
    if (! pimpl->wii_.findController()) {
      return;
    }
#endif

    // !!! スレッド化すべきかも
    // !!! あと、戻り値に応じて、エラー情報を出力すべき
    if (! pimpl->wii_.connect()) {
      return;
    }

    // 押されたとき、connect()
    connect_button_->setText(tr("Disconnect"));

    pimpl->redraw_timer_->start(100);

  } else {
    // 離されたとき、disconnect()
    connect_button_->setText(tr("Connect"));
    pimpl->wii_.disconnect();

    pimpl->redraw_timer_->stop();
    pimpl->clearForm(this);
  }
}


void WiiPoseWidget::updatePose(void) {

  // 加速度情報の取得
  Grid3D<double> acc;
  pimpl->wii_.getAcceleration(acc);

  // 描画情報の更新
  double length = sqrt((acc.x * acc.x) + (acc.z * acc.z));
  double x_rad = atan2(-acc.y, length);
  double z_rad = atan2(acc.z, acc.x);
  int rotate_x = static_cast<int>(180 * x_rad / M_PI);
  int rotate_z = static_cast<int>(180 * z_rad / M_PI);

  // メッセージの更新
  size_t battery_percent = pimpl->wii_.getBatteryPercent();
  battery_label_->setText(QString(tr("battery %1 [%]")).arg(battery_percent));

  pimpl->paint_widget_->rotate(rotate_x, 0, rotate_z);
}
