/*!
  \file
  \brief 二次元データ表示

  \author Satofumi KAMIMURA

  $Id: XyGraphWidget.cpp 270 2008-10-13 21:55:10Z satofumi $
*/

#include "XyGraphWidget.h"
#include <deque>

namespace
{
  const double DrawLengthMax = 5.3;

  struct XyData {
    int timestamp;
    double x;
    double y;

    XyData(int timestamp_, double x_, double y_)
      : timestamp(timestamp_), x(x_), y(y_)
    {
    }
  };
};


struct XyGraphWidget::pImpl
{
  enum {
    MinimumWidth = 100,
    MinimumHeight = 100,

    DefaultDrawPeriod = 2000,   // [msec]
  };

  XyGraphWidget* parent_;
  std::deque<XyData> xy_data_;
  int draw_period_;

  QColor clear_color_;


  pImpl(XyGraphWidget* parent)
    : parent_(parent), draw_period_(DefaultDrawPeriod), clear_color_(Qt::gray)
  {
  }


  void initializeForm(void)
  {
    parent_->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    parent_->setMinimumSize(MinimumWidth, MinimumHeight);
  }


  void initializeGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);

    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }


  void resizeGL(int width, int height)
  {
    glViewport(0, 0, width, height);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    double aspect = 1.0 * width / height;
    glOrtho(-1.0 * aspect, +1.0 * aspect, -1.0, +1.0,
            std::numeric_limits<int>::min(), std::numeric_limits<int>::max());

    glMatrixMode(GL_MODELVIEW);
  }


  void paintGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glLoadIdentity();

    // 軸の描画
    drawAxis();

    // データの描画
    if (xy_data_.empty()) {
      return;
    }
    drawData();
  }


  void drawAxis(void)
  {
    glColor3d(1.0, 1.0, 1.0);
    glBegin(GL_LINES);
    glVertex2d(-0.95, 0.0);
    glVertex2d(+0.95, 0.0);

    glVertex2d(0.0, -0.95);
    glVertex2d(0.0, +0.95);
    glEnd();
  }


  void drawData(void)
  {
    glPointSize(1.6);
    //glBegin(GL_POINTS);
    glBegin(GL_LINE_STRIP);
    int current_timestamp = xy_data_.back().timestamp;
    for (std::deque<XyData>::iterator it = xy_data_.begin();
         it != xy_data_.end(); ++it) {
      double alpha = 1.0 *
        (draw_period_ - (current_timestamp - it->timestamp)) / draw_period_;
      glColor4d(0.0, 0.0, 1.0, alpha);
      double x = it->x / DrawLengthMax;
      double y = it->y / DrawLengthMax;
      glVertex2d(x, y);
    }
    glEnd();
  }


  void removeOldData(int current_timestamp)
  {
    while (! xy_data_.empty()) {
      int timestamp = xy_data_.front().timestamp;
      if ((timestamp + draw_period_) > current_timestamp) {
        break;
      }
      xy_data_.pop_front();
    }
  }
};


XyGraphWidget::XyGraphWidget(QWidget* parent)
  : QGLWidget(parent), pimpl(new pImpl(this))
{
  pimpl->initializeForm();
}


XyGraphWidget::~XyGraphWidget(void)
{
}


void XyGraphWidget::clear(void)
{
  pimpl->xy_data_.clear();
}


void XyGraphWidget::redraw(void)
{
  updateGL();
}


void XyGraphWidget::setDrawPeriod(int msec)
{
  pimpl->draw_period_ = msec;
}


void XyGraphWidget::setData(int timestamp, double x, double y)
{
  pimpl->xy_data_.push_back(XyData(timestamp, x, y));

  // 古いデータの破棄
  pimpl->removeOldData(timestamp);
}


void XyGraphWidget::initializeGL(void)
{
  pimpl->initializeGL();
}


void XyGraphWidget::resizeGL(int width, int height)
{
  pimpl->resizeGL(width, height);
}


void XyGraphWidget::paintGL(void)
{
  pimpl->paintGL();
}
