/*!
  \file
  \brief Wii ブレード用のウィンドウ管理

  \author Satofumi KAMIMURA

  $Id: WiiBladeWindow.cpp 260 2008-10-08 10:39:09Z satofumi $

  \todo 記録中もスライドバーを移動させるか、を検討する
*/

#include "WiiBladeWindow.h"
#include "SwingImageWidget.h"
#include "TimeGraphWidget.h"
#include "XyGraphWidget.h"
#include "WiiJoystick.h"
#include "StopWatch.h"
#include "getTicks.h"
#include "AngleTypes.h"
#include "MathUtils.h"
#include "split.h"
#ifdef Q_WS_WIN
#include "rescanWiimote.h"
#endif
#include <QSettings>
#include <QTimer>
#include <QShortcut>
#include <QMessageBox>
#include <QSysInfo>
extern "C"
{
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
}
#include <luabind/luabind.hpp>
#include <boost/lexical_cast.hpp>
#include <fstream>

using namespace qrk;
using namespace boost;

namespace
{
  const char* Organization = "Hyakuren Soft";
  const char* Application = "Wii Blade";

  const char* RecordFile = "acceleration.txt";

  enum {
    RedrawInterval = 50,       // [msec]
  };
};


struct WiiBladeWindow::pImpl
{
  enum {
    CaptureInterval = 10,       // [msec]
  };

  WiiBladeWindow* parent_;
  SwingImageWidget swing_image_;
  TimeGraphWidget time_graph_;
  XyGraphWidget xy_graph_;

  WiiJoystick wii_;

  int blade_degree_;
  QTimer capture_timer_;
  QTimer play_timer_;
  StopWatch play_stop_watch_;
  std::ofstream* record_fout_;
  std::ifstream* play_fin_;
  std::string last_read_line_;
  bool now_playing_;
  bool repeat_mode_;
  size_t ticks_max_;

  int last_draw_ticks_;

  lua_State* lua_;


  pImpl(WiiBladeWindow* parent)
    : parent_(parent),
      swing_image_(parent_), time_graph_(parent_), xy_graph_(parent_),
      blade_degree_(0), record_fout_(NULL), play_fin_(NULL),
      now_playing_(false), repeat_mode_(false), ticks_max_(0),
      last_draw_ticks_(0), lua_(lua_open())
  {
    luaL_openlibs(lua_);
    luabind::open(lua_);

    luabind::module(lua_) [
      luabind::def("sin", &sin),
      luabind::def("cos", &cos),
      luabind::def("atan2", &atan2),
      luabind::def("sqrt", &sqrt)
    ];
  }


  ~pImpl(void)
  {
    lua_close(lua_);

    delete record_fout_;
    delete play_fin_;
  }


  void saveSettings(void)
  {
    QSettings settings(Organization, Application);

    settings.setValue("geometry", parent_->saveGeometry());
    settings.setValue("repeat_mode", parent_->repeat_button_->isChecked());
  }


  void loadSettings(void)
  {
    QSettings settings(Organization, Application);

    parent_->restoreGeometry(settings.value("geometry").toByteArray());
    repeat_mode_ = settings.value("repeat_mode", true).toBool();
    parent_->repeat_button_->setChecked(repeat_mode_);
  }


  void initializeForm(void)
  {
    // ウィジット
    parent_->dummy_label_1_->hide();
    parent_->swingImage_layout_->addWidget(&swing_image_);

    parent_->dummy_label_2_->hide();
    parent_->timeGraph_layout_->addWidget(&time_graph_);

    parent_->dummy_label_3_->hide();
    parent_->xyGraph_layout_->addWidget(&xy_graph_);

    time_graph_.setDataColor(0, Qt::blue);
    time_graph_.setDataColor(1, Qt::red);
    time_graph_.setDataColor(2, Qt::green);

    // メニュー項目
    connect(parent_->action_quit_, SIGNAL(triggered()),
            parent_, SLOT(close()));
    connect(parent_->action_about_, SIGNAL(triggered()),
            parent_, SLOT(aboutApplication()));

    // フォーム
    connect(parent_->rescan_button_, SIGNAL(clicked()),
            parent_, SLOT(rescanPressed()));
    connect(parent_->connect_button_, SIGNAL(clicked()),
            parent_, SLOT(connectPressed()));
    connect(parent_->disconnect_button_, SIGNAL(clicked()),
            parent_, SLOT(disconnectPressed()));
    connect(parent_->play_button_, SIGNAL(clicked()),
            parent_, SLOT(playPressed()));
    connect(parent_->repeat_button_, SIGNAL(clicked(bool)),
            parent_, SLOT(repeatPressed(bool)));

    // データ取得
    connect(&capture_timer_, SIGNAL(timeout()),
            parent_, SLOT(recordData()));
    connect(&play_timer_, SIGNAL(timeout()),
            parent_, SLOT(updatePlayData()));

    // キーバインド
    (void) new QShortcut(Qt::Key_Space, parent_, SLOT(playPressed()));
  }


  void rescanPressed(void)
  {
#ifdef Q_WS_WIN
    // Windows ならば、WiimoteRescanner を実行する
    rescanWiimote();
#endif
    // !!! Linux ならば、"hciconfig hci0 up" を実行べき
    // !!! ただし、要、管理者権限
  }


  void connectPressed(void)
  {
    // !!! スレッドで接続処理を行い、終了シグナルで処理を行うべき
    // !!! スレッド処理中は、接続を disable にすること
    // !!! また、接続中は、その旨を示すメッセージボックスを表示すること

    if (! wii_.connect()) {
      QMessageBox::warning(parent_, "connect",
                           wii_.what(), QMessageBox::Close);
      return;
    }

    // データ取得の開始
    setConnectEnabled(false);
    delete record_fout_;
    record_fout_ = new std::ofstream(RecordFile);

    startPlay();
    capture_timer_.start(pImpl::CaptureInterval);
  }


  void disconnectPressed(void)
  {
    stopPlay();
    capture_timer_.stop();
    wii_.disconnect();
    setConnectEnabled(true);
  }


  void setConnectEnabled(bool enable)
  {
    parent_->connect_button_->setEnabled(enable);
    parent_->rescan_button_->setEnabled(! enable);
    parent_->disconnect_button_->setEnabled(! enable);

    setDataEnabled(! enable);
  }


  void setDataEnabled(bool enable)
  {
    parent_->acceleration_group_->setEnabled(enable);
  }


  void recordData(void)
  {
    Grid3D<double> acc;
    wii_.getAcceleration(acc);
    int timestamp = getTicks();

    *record_fout_ << timestamp
                  << '\t' << acc.x << '\t' << acc.y << '\t' << acc.z
                  << std::endl;

    setNewAccelerationData(acc, timestamp);
  }


  void updateAccelerationValue(const Grid3D<double>& acc)
  {
    parent_->x_acc_label_->setText(doubleToString(acc.x).c_str());
    parent_->y_acc_label_->setText(doubleToString(acc.y).c_str());
    parent_->z_acc_label_->setText(doubleToString(acc.z).c_str());
  }


  std::string doubleToString(double value)
  {
    char buffer[] = "-0.0000";
    snprintf(buffer, sizeof(buffer), "%.02f", value);

    return buffer;
  }


  void startPlay(void)
  {
    if (luaL_dofile(lua_, "calculateAngle.lua") != 0) {
      printf("error: %s\n", lua_tostring(lua_, -1));
    }

    xy_graph_.clear();
    time_graph_.clear();
    swing_image_.clear();
    setDataEnabled(true);
  }


  void stopPlay(void)
  {
    play_timer_.stop();
    last_draw_ticks_ = 0;

    if (capture_timer_.isActive()) {
      // 記録時間の退避
      ticks_max_ = play_stop_watch_.getTicks();
    }
    play_stop_watch_.stop();

    now_playing_ = false;
    setPlayingIcon(false);
    updatePlayEnable();
    setDataEnabled(false);
  }


  void pausePlay(void)
  {
    play_stop_watch_.pause();
  }


  void resumePlay(void)
  {
    play_stop_watch_.resume();
  }


  void updatePlayData(void)
  {
    if (last_read_line_.empty()) {
      if (! getline(*play_fin_, last_read_line_)) {
        last_read_line_.clear();
      }
    }

    // 取得データのパース
    std::vector<std::string> tokens;
    if (split(tokens, last_read_line_, "\t") != 4) {
      // 異常なデータに遭遇したら、再生を中断する
      stopPlay();

      if (repeat_mode_) {
        // 繰り返し再生する
        parent_->playPressed();
      }
      return;
    }
    int target_ticks = atoi(tokens[0].c_str());

    // 再生タイミングを経過していたら出力を更新する
    int current_ticks = play_stop_watch_.getTicks();
    if (current_ticks > target_ticks) {
      last_read_line_.clear();

      Grid3D<double> acc(atof(tokens[1].c_str()),
                         atof(tokens[2].c_str()),
                         atof(tokens[3].c_str()));

      setNewAccelerationData(acc, target_ticks);
    }
  }


  void setNewAccelerationData(const Grid3D<double>& acc, int timestamp)
  {
    updateAccelerationValue(acc);

    // X-Y グラフの更新
    xy_graph_.setData(timestamp, acc.x, acc.y);

    // Time グラフの更新
    time_graph_.setTime(timestamp);
    time_graph_.setData(0, acc.x);
    time_graph_.setData(1, acc.y);
    time_graph_.setData(2, acc.z);

    // スライダ位置の更新
    if (now_playing_) {
      int percent = static_cast<int>(100.0 * timestamp / ticks_max_);
      parent_->progress_slider_->setValue(percent);
    } else {
      // !!!
    }

    blade_degree_ = luabind::call_function<int>(lua_, "calculateAngle",
                                                timestamp, acc.x, acc.y, acc.z);

    if ((timestamp - last_draw_ticks_) > RedrawInterval) {
      last_draw_ticks_ = timestamp;
      redraw();
    }
  }


  void redraw(void)
  {
    xy_graph_.redraw();
    time_graph_.redraw();
    swing_image_.setBladeAngle(deg(blade_degree_));
  }



  bool openRecordFile(void)
  {
    if (play_fin_ && (play_fin_->is_open() && (! play_fin_->eof()))) {
      // 既に開いている場合は、処理を戻す
      return true;
    }

    // 新規に開く
    delete play_fin_;
    play_fin_ = new std::ifstream(RecordFile);
    if (! play_fin_->is_open()) {
      return false;
    }

    // 再生時間の取得
    if (ticks_max_ == 0) {
      size_t ticks_max = 0;
      std::string line;
      while (getline(*play_fin_, line)) {
        size_t ticks = atoi(line.c_str());
        if (ticks > ticks_max) {
          ticks_max = ticks;
        }
      }
      if (ticks_max > 0) {
        ticks_max_ = ticks_max;

        // play_fin_ が無効になっているため、再度ファイルを開く
        return openRecordFile();
      }
    }

    return true;
  }


  void setPlayEnabled(bool enable)
  {
    parent_->play_button_->setEnabled(enable);
    parent_->repeat_button_->setEnabled(enable);
    parent_->progress_slider_->setEnabled(enable);
  }


  void updatePlayEnable(void)
  {
    bool play_enable = openRecordFile();
    setPlayEnabled(play_enable);
  }


  void setPlayingIcon(bool playing)
  {
    if (playing) {
      parent_->play_button_->setIcon(QPixmap(":/icons/pause_icon"));
    } else {
      parent_->play_button_->setIcon(QPixmap(":/icons/play_icon"));
    }
  }


  void startPlayTimer(void)
  {
    play_stop_watch_.start();
    play_timer_.start(10);
  }


  void playPressed(void)
  {
    if (! now_playing_) {
      if (play_stop_watch_.isPause()) {
        // 一時停止の処理を再開
        resumePlay();

      } else {
        // 再生開始
        startPlay();
        startPlayTimer();
      }

    } else {
      // 一時停止
      pausePlay();
    }

    now_playing_ = ! now_playing_;
    setPlayingIcon(now_playing_);
  }


  void repeatPressed(bool checked)
  {
    repeat_mode_ = checked;
  }
};


WiiBladeWindow::WiiBladeWindow(void) : QMainWindow(), pimpl(new pImpl(this))
{
  setupUi(this);
  pimpl->initializeForm();
  pimpl->loadSettings();
  pimpl->updatePlayEnable();
}


WiiBladeWindow::~WiiBladeWindow(void)
{
  pimpl->saveSettings();
}


void WiiBladeWindow::aboutApplication(void)
{
  QMessageBox::about(this, tr("About WiiBlade"),
                     tr("<h2>Wii Blade ($Rev: 260 $)</h2>"
                        "<p>Demo Application for Wii controller</p>"
                        "<p>Report bugs to "
                        "&lt;satofumi@users.sourceforge.jp&gt;</p>"));
}


void WiiBladeWindow::rescanPressed(void)
{
  pimpl->rescanPressed();
}


void WiiBladeWindow::connectPressed(void)
{
  pimpl->connectPressed();
}


void WiiBladeWindow::disconnectPressed(void)
{
  pimpl->disconnectPressed();
}


void WiiBladeWindow::recordData(void)
{
  pimpl->recordData();
}


void WiiBladeWindow::playPressed(void)
{
  pimpl->playPressed();
}


void WiiBladeWindow::repeatPressed(bool checked)
{
  pimpl->repeatPressed(checked);
}


void WiiBladeWindow::updatePlayData(void)
{
  pimpl->updatePlayData();
}
