/*!
  \file
  \brief 時刻データ表示

  \author Satofumi KAMIMURA

  $Id: TimeGraphWidget.cpp 247 2008-09-30 22:22:58Z satofumi $

  \todo 点列描画を、連続直線での描画に変更する
*/

#include "TimeGraphWidget.h"
#include <deque>

namespace
{
  const double DrawHeightMax = 5.0;
  const int DrawWidthMax = 5000;

  struct TimeData
  {
    int timestamp;
    std::vector<double> data;
  };


  struct Color
  {
    double r;
    double g;
    double b;
  };
};


struct TimeGraphWidget::pImpl
{
  enum {
    MinimumWidth = 100,
    MinimumHeight = 100,

    DefaultDrawPeriod = 10000,   // [msec]
  };

  TimeGraphWidget* parent_;
  int draw_period_;

  QColor clear_color_;

  int max_index_;
  std::vector<Color> plot_colors_;
  std::deque<TimeData> time_data_;

  int current_timestamp_;


  pImpl(TimeGraphWidget* parent)
    : parent_(parent), draw_period_(DefaultDrawPeriod), clear_color_(Qt::gray),
      max_index_(0), current_timestamp_(0)
  {
  }


  void initializeForm(void)
  {
    parent_->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
  }


  void initializeGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);

    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }


  void resizeGL(int width, int height)
  {
    glViewport(0, 0, width, height);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    double aspect = 1.0;
    glOrtho(-1.0 * aspect, +1.0 * aspect, -1.0, +1.0,
            std::numeric_limits<int>::min(), std::numeric_limits<int>::max());

    glMatrixMode(GL_MODELVIEW);
  }


  void paintGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glLoadIdentity();

    // 軸の描画
    drawAxis();

    // データの描画
    if (time_data_.empty()) {
      return;
    }
    drawData();
  }


  void drawAxis(void)
  {
    glColor3d(1.0, 1.0, 1.0);
    glBegin(GL_LINES);
    glVertex2d(-0.95, 0.0);
    glVertex2d(+0.95, 0.0);

    glVertex2d(0.90, -0.95);
    glVertex2d(0.90, +0.95);
    glEnd();
  }


  void drawData(void)
  {
    glBegin(GL_POINTS);
    for (std::deque<TimeData>::iterator time_it = time_data_.begin();
         time_it != time_data_.end(); ++time_it) {

      double alpha = 1.0 *
        (draw_period_ - (current_timestamp_ - time_it->timestamp))
        / draw_period_;

      // !!! Y 軸の描画位置を示す 0.95 を変数に置き換えるべき
      double x =
        0.95 - (1.0 * (current_timestamp_ - time_it->timestamp) / DrawWidthMax);

      int index = 0;
      std::vector<double>::iterator end_data_it = time_it->data.end();
      for (std::vector<double>::iterator data_it = time_it->data.begin();
           data_it != end_data_it; ++data_it, ++index) {

        glColor4d(plot_colors_[index].r,
                  plot_colors_[index].g,
                  plot_colors_[index].b,
                  alpha);

        double y = *data_it / DrawHeightMax;

        glVertex2d(x, y);
      }
    }
    glEnd();
  }
};


TimeGraphWidget::TimeGraphWidget(QWidget* parent)
  : QGLWidget(parent), pimpl(new pImpl(this))
{
  pimpl->initializeForm();
}


TimeGraphWidget::~TimeGraphWidget(void)
{
}


void TimeGraphWidget::clear(void)
{
  pimpl->time_data_.clear();
}


void TimeGraphWidget::redraw(void)
{
  updateGL();
}


void TimeGraphWidget::setDrawPeriod(int msec)
{
  pimpl->draw_period_ = msec;
}


void TimeGraphWidget::setDataColor(int index, Qt::GlobalColor color)
{
  if (index >= pimpl->max_index_) {
    pimpl->max_index_ = index;
    pimpl->plot_colors_.resize(index + 1);
  }

  QColor q_color(color);
  pimpl->plot_colors_[index].r = q_color.red();
  pimpl->plot_colors_[index].g = q_color.green();
  pimpl->plot_colors_[index].b = q_color.blue();
}


void TimeGraphWidget::setTime(int timestamp)
{
  pimpl->current_timestamp_ = timestamp;

  TimeData time_data;
  time_data.timestamp = timestamp;
  time_data.data.resize(pimpl->max_index_ + 1);
  pimpl->time_data_.push_back(time_data);
}


void TimeGraphWidget::setData(int index, double value)
{
  TimeData& time_data = pimpl->time_data_.back();
  time_data.data[index] = value;
}


void TimeGraphWidget::initializeGL(void)
{
  pimpl->initializeGL();
}


void TimeGraphWidget::resizeGL(int width, int height)
{
  pimpl->resizeGL(width, height);
}


void TimeGraphWidget::paintGL(void)
{
  pimpl->paintGL();
}
