/*!
  \file
  \brief 素振りのイメージ描画

  \author Satofumi KAMIMURA

  $Id: SwingImageWidget.cpp 239 2008-09-26 21:35:40Z satofumi $
*/

#include "SwingImageWidget.h"
#include "AngleTypes.h"
#include "MathUtils.h"

using namespace qrk;


struct SwingImageWidget::pImpl
{
  enum {
    MinimumWidth = 100,
    MinimumHeight = 100,
  };

  SwingImageWidget* parent_;
  QColor clear_color_;
  Angle blade_angle_;


  pImpl(SwingImageWidget* parent) : parent_(parent), clear_color_(Qt::gray)
  {
  }


  void initializeForm(void)
  {
    parent_->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    parent_->setMinimumSize(MinimumWidth, MinimumHeight);
  }


  void initializeGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glEnable(GL_TEXTURE_2D);
  }


  void resizeGL(int width, int height)
  {
    glViewport(0, 0, width, height);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    double aspect = 1.0 * width / height;
    enum { DefaultViewLength = 2000 };
    glOrtho(-DefaultViewLength * aspect, +DefaultViewLength * aspect,
            -DefaultViewLength, DefaultViewLength,
            std::numeric_limits<int>::min(), std::numeric_limits<int>::max());

    glMatrixMode(GL_MODELVIEW);
  }


  void paintGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glLoadIdentity();

    // データの描画
    glColor3d(0.0, 1.0, 0.0);
    glBegin(GL_LINES);
    double radian = blade_angle_.to_rad();
    double x = 1000 * cos(radian);
    double y = 1000 * sin(radian);
    glVertex2d(0.0, 0.0);
    glVertex2d(x, y);
    glEnd();
  }
};


SwingImageWidget::SwingImageWidget(QWidget* parent)
  : QGLWidget(parent), pimpl(new pImpl(this))
{
  pimpl->initializeForm();
}


SwingImageWidget::~SwingImageWidget(void)
{
}


void SwingImageWidget::clear(void)
{
  pimpl->blade_angle_ = deg(0);
}


void SwingImageWidget::setBladeAngle(const Angle& angle)
{
  pimpl->blade_angle_ = angle;
  updateGL();
}


void SwingImageWidget::initializeGL(void)
{
  pimpl->initializeGL();
}


void SwingImageWidget::resizeGL(int width, int height)
{
  pimpl->resizeGL(width, height);
}


void SwingImageWidget::paintGL(void)
{
  pimpl->paintGL();
}
