/*!
  \file
  \brief モータの動作スレッド

  \author Satofumi KAMIMURA

  $Id: SwingThread.cpp 253 2008-10-02 06:42:39Z satofumi $

  初期位置が水平でないため、HorizontalOffset 位置をゼロとして扱う

  \todo モータスピードを可変にする
*/

#include "SwingThread.h"
#include "DensanMotor.h"
#include "AngleTypes.h"
#include <QMutex>
#include <boost/integer_traits.hpp>

using namespace qrk;


struct SwingThread::pImpl
{
  enum {
    DefaultMotorSpeed = 1,      // [percent]
    WaitMsec = 500,            // コマンド発行間隔
    HorizontalOffset = 0,      // [deg]
    InvalidCounter = boost::integer_traits<int>::const_max,
  };

  DensanMotor* motor_;
  QMutex mutex_;
  int initial_counter_;
  int counter_;
  SwingState state_;
  Angle swing_range_;


  pImpl(DensanMotor* motor)
    : motor_(motor), initial_counter_(InvalidCounter), counter_(0),
      state_(Initializing), swing_range_(deg(90))
  {
  }


  void swing(void)
  {
    state_ = Initializing;

#if 0
    // 初期位置への移動
    while (! motor_->initialize()) {
      msleep(100);
    }

    // 0 度の位置まで移動
    moveTo(HorizontalOffset);
    moveTo(HorizontalOffset);
#endif
    mutex_.lock();
    //initial_counter_ = motor_->counter();
    initial_counter_ = 0;
    mutex_.unlock();
    state_ = CW;

    while (1) {
      // 指定の位置まで移動
      moveTo(swing_range_.to_deg() + HorizontalOffset);
      state_ = CW;

      // 0 度の位置まで移動
      moveTo(HorizontalOffset);
      state_ = CCW;
    }
  }


  void moveTo(int degree, int speed_percent = DefaultMotorSpeed)
  {
    while (1) {
      msleep(WaitMsec);

      mutex_.lock();
      if (motor_->move(deg(degree), speed_percent)) {
        mutex_.unlock();
        break;
      }
      mutex_.unlock();
    }
  }


  int counter(void)
  {
    if (initial_counter_ == InvalidCounter) {
      return -1;
    }

    mutex_.lock();
    int current_counter = motor_->counter() - initial_counter_;
    mutex_.unlock();

    return current_counter;
  }
};


SwingThread::SwingThread(DensanMotor* motor) : pimpl(new pImpl(motor))
{
}


SwingThread::~SwingThread(void)
{
}


qrk::Angle SwingThread::swingRange(void)
{
  return pimpl->swing_range_;
}


//! \attention 排他制御はしていない。停止中に設定すること
void SwingThread::setSwingRange(const Angle& angle)
{
  pimpl->swing_range_ = angle;
}


void SwingThread::run(void)
{
  pimpl->swing();
}


int SwingThread::counter(void)
{
  return pimpl->counter();
}


SwingThread::SwingState SwingThread::state(void)
{
  pimpl->mutex_.lock();
  SwingState current_state = pimpl->state_;
  pimpl->mutex_.unlock();

  return current_state;
}


void SwingThread::stop(void)
{
  pimpl->mutex_.lock();
  pimpl->motor_->stop();
  pimpl->mutex_.unlock();
}
