/*!
  \file
  \brief メインウィンドウ

  操作 GUI と、２画面のデータ表示ウィジットを管理する

  \author Satofumi KAMIMURA

  $Id: SwingScan3DWindow.cpp 253 2008-10-02 06:42:39Z satofumi $

  \todo disconnect したときに、再開できないのをなんとかする
  \todo ポートを重複して選択できないようにする
  \todo 開いてある子ウィンドウを閉じるようにする
*/

#include "SwingScan3DWindow.h"
#include "ConfigDialog.h"
#include "SwingThread.h"
#include "ScanedControlWidget.h"
#include "UrgCtrl.h"
#include "UrgUsbCom.h"
#include "convert2DWithIntensity.h"
#include "DensanMotor.h"
#include "FindComPorts.h"
#include "MathUtils.h"
#include "AngleTypes.h"
#include "ConvertStdStringPath.h"
#include <QFileDialog>
#include <QSettings>
#include <QMessageBox>
#include <QTimer>
#include <QMutex>

using namespace qrk;


struct SwingScan3DWindow::pImpl
{
  enum {
    RedrawMsec = 133,
    DrawWindowsNum = 2,
  };

  SwingScan3DWindow* parent_;
  ConfigDialog config_;

  UrgCtrl urg_;
  DensanMotor motor_;

  SwingThread swing_thread_;
  QTimer capture_timer_;
  QTimer redraw_timer_;

  int urg_scan_msec_;
  int urg_min_distance_;
  int urg_max_distance_;
  int urg_data_max_;
  long* urg_data_;
  long* intensity_data_;

  QMutex draw_mutex_;
  SwingThread::SwingState state_;
  Points3D* points_buffers_[DrawWindowsNum];
  Intensity3D* intensity_buffers_[DrawWindowsNum];
  ScanedControlWidget draw_widgets_[DrawWindowsNum];

  int swing_offset_;


  pImpl(SwingScan3DWindow* parent)
    : parent_(parent), config_(parent_), swing_thread_(&motor_),
      urg_scan_msec_(100), urg_min_distance_(20), urg_max_distance_(0),
      urg_data_max_(1), urg_data_(NULL), intensity_data_(NULL),
      state_(SwingThread::Initializing), swing_offset_(0)
  {
    for (int i = 0; i < DrawWindowsNum; ++i) {
      points_buffers_[i] = new Points3D;
      intensity_buffers_[i] = new Intensity3D;
    }
  }


  ~pImpl(void)
  {
    stopCapture();
    delete [] urg_data_;
    delete [] intensity_data_;

    for (int i = 0; i < DrawWindowsNum; ++i) {
      delete points_buffers_[i];
      delete intensity_buffers_[i];
    }

    saveSettings();
  };


  void saveSettings(void)
  {
    QSettings settings("Hokuyo LTD.", "SwingScan3D");

    // ウィンドウの位置と場所
    QRect position = parent_->geometry();
    QSize size = parent_->size();
    settings.setValue("geometry", QRect(position.x(), position.y(),
                                        size.width(), size.height()));

    // 拡大率
    settings.setValue("zoom_0", draw_widgets_[0].zoomPercent());
    settings.setValue("zoom_1", draw_widgets_[1].zoomPercent());

    // 設定
    settings.setValue("swing_range", swing_thread_.swingRange().to_deg());
    settings.setValue("swing_offset", swing_offset_);
    settings.setValue("ignore_length", urg_min_distance_);
  }


  void loadSettings(void)
  {
    QSettings settings("Hokuyo LTD.", "SwingScan3D");

    // ウィンドウの位置と場所
    QRect position = settings.value("geometry", parent_->geometry()).toRect();
    parent_->setGeometry(position);

    // 拡大率
    draw_widgets_[0].setZoomPercent(settings.value("zoom_0", 20).toInt());
    draw_widgets_[1].setZoomPercent(settings.value("zoom_0", 20).toInt());

    // 設定
    config_.setSwingRange(deg(settings.value("swing_range", 90).toInt()));
    config_.setSwingOffset(deg(settings.value("swing_offset", 0).toInt()));
    config_.setIgnoreLength(settings.value("ignore_length", 20).toInt());
  }


  void initializeForm(void)
  {
    // 描画ウィジットの配置
    parent_->window_1_dummy_->hide();
    parent_->window_2_dummy_->hide();
    for (int i = 0; i < DrawWindowsNum; ++i) {
      parent_->main_layout_->addWidget(&draw_widgets_[i]);
    }

    // メニューまわり
    connect(parent_->action_quit_, SIGNAL(triggered()), parent_, SLOT(close()));
    connect(parent_->action_save_, SIGNAL(triggered()), parent_, SLOT(save()));
    rescan();

    // 接続まわり
    parent_->urg_combobox_->setSizeAdjustPolicy(QComboBox::AdjustToContents);
    parent_->motor_combobox_->setSizeAdjustPolicy(QComboBox::AdjustToContents);
    connect(parent_->rescan_button_, SIGNAL(clicked()),
            parent_, SLOT(rescan()));
    connect(parent_->connect_button_, SIGNAL(clicked()),
            parent_, SLOT(connectSensors()));
    connect(parent_->disconnect_button_, SIGNAL(clicked()),
            parent_, SLOT(disconnectSensors()));
    connect(&capture_timer_, SIGNAL(timeout()), parent_, SLOT(capture()));

    // 設定まわり
    connect(parent_->config_button_, SIGNAL(clicked()),
            parent_, SLOT(config()));

    // 再描画まわり
    connect(&redraw_timer_, SIGNAL(timeout()), parent_, SLOT(redraw()));
    redraw_timer_.start(RedrawMsec);

    // Config 設定更新の受け取りまわり
    connect(&config_, SIGNAL(swingRangeChanged(int)),
            parent_, SLOT(swingRange(int)));
    connect(&config_, SIGNAL(swingOffsetChanged(int)),
            parent_, SLOT(swingOffset(int)));
    connect(&config_, SIGNAL(ignoreLengthChanged(int)),
            parent_, SLOT(ignoreLength(int)));

    // モータ制御まわり
    connect(&config_, SIGNAL(initializePositionOrder()),
	    parent_, SLOT(initializePositionOrder()));

    // 前回設定の読み出し
    loadSettings();
  }


  void rescan(void)
  {
    // URG ポートの更新
    UrgUsbCom urg_usb;
    FindComPorts urg_finder(&urg_usb);
    std::vector<std::string> urg_ports = urg_finder.find();
    updateCombobox(parent_->urg_combobox_, urg_ports);

    // モータポートの更新
    FindComPorts motor_finder(&motor_);
    std::vector<std::string> motor_ports = motor_finder.find();
    updateCombobox(parent_->motor_combobox_, motor_ports);

    // ポートがなければ、接続させない
    parent_->connect_button_->setEnabled(! (urg_ports.empty() ||
                                            motor_ports.empty()));
  }


  void updateCombobox(QComboBox* combobox,
                      const std::vector<std::string>& ports)
  {
    combobox->clear();
    for (std::vector<std::string>::const_iterator it = ports.begin();
         it != ports.end(); ++it) {
      combobox->addItem(it->c_str());
    }
  }


  void urgSelected(void)
  {
    // 同じポートが選択されないよう、motor 側を操作する
    // !!!
  }


  void motorSelected(void)
  {
    // 同じポートが選択されないよう、urg 側を操作する
    // !!!
  }


  void connectSensors(void)
  {
    // URG との接続
    const std::string urg_device =
      parent_->urg_combobox_->currentText().toStdString();
    if (! urg_.connect(urg_device.c_str())) {
      QMessageBox::warning(parent_, "URG", urg_.what(), QMessageBox::Close);
      return;
    }
    updateUrgSettings();

    // モータとの接続
    const std::string motor_device =
      parent_->motor_combobox_->currentText().toStdString();
    if ((! motor_.connect(motor_device.c_str())) && (motor_.version())) {
      // !!! MessageBox でエラー内容を出力
      QMessageBox::warning(parent_, "DensanMotor",
                           "DensanMotor::connect: fail.", QMessageBox::Close);

      // !!! 自動で解放されるようにしたい
      urg_.disconnect();
      return;
    }

    // 更新処理の動作開始
    startCapture();
  }


  void disconnectSensors(void)
  {
    stopCapture();

    urg_.disconnect();
    motor_.disconnect();
  }


  void startCapture(void)
  {
    swing_thread_.start();
    capture_timer_.start(urg_scan_msec_);
    setStartEnabled(false);
  }


  void stopCapture(void)
  {
    swing_thread_.terminate();
    motor_.stop();
    capture_timer_.stop();
    setStartEnabled(true);
  }


  void setStartEnabled(bool enable)
  {
    parent_->connect_button_->setEnabled(enable);
    parent_->disconnect_button_->setEnabled(! enable);
  }


  void updateUrgSettings(void)
  {
    // 取得データを間引く
    urg_.setSkipLines(2);

    //urg_.setCaptureMode(AutoCapture);
    urg_.setCaptureMode(IntensityCapture);

    urg_scan_msec_ = urg_.scanMsec();
    //urg_min_distance_ = urg_.minDistance();
    urg_max_distance_ = urg_.maxDistance();

    urg_data_max_ = urg_.maxBufferSize();
    if (urg_data_) {
      delete [] urg_data_;
    }
    if (intensity_data_) {
      delete [] intensity_data_;
    }
    urg_data_ = new long [urg_data_max_];
    intensity_data_ = new long [urg_data_max_];
  }


  void capture(void)
  {
    // 移動状態が変更したら、データ追加、データ表示を行うウィンドウを切替える
    SwingThread::SwingState current_state = swing_thread_.state();
    updateDrawState(current_state);

    // 現在姿勢の取得
    // 90 度位置のカウンタ値を 32768 とみなす
    double angle_radian = M_PI / 2.0 * swing_thread_.counter() / 32768.0;
    if (angle_radian > (135.0 / 18.0 * M_PI)) {
      swing_thread_.stop();
    }

    // 取り付けオフセットのための調整
    angle_radian += (M_PI / 2.0) * swing_offset_ / 90.0;

    // 0 度の位置を描画の正面にするための調整
    angle_radian -= (M_PI / 2.0);

    // URG データの取得
    int n = urg_.capture(urg_data_, urg_data_max_);
    if (n < 0) {
      return;
    }

    // 3D 位置に変換して格納
    Grid<int> position_offset(70, 20);
    PointsLine line;
    int intensity_n = urg_.intensity(intensity_data_, urg_data_max_);
    if (n == intensity_n) {
      IntensityLine intensity_line;
      convertUrgData(line, angle_radian,
                     urg_data_, n, position_offset,
                     intensity_data_, intensity_line);
      draw_mutex_.lock();
      points_buffers_[0]->push_front(line);

      intensity_buffers_[0]->push_front(intensity_line);
    }
    draw_mutex_.unlock();
  }


  void updateDrawState(SwingThread::SwingState current_state)
  {
    draw_mutex_.lock();
    if (current_state == SwingThread::Initializing) {
      // 保持データのクリア
      for (int i = 0; i < DrawWindowsNum; ++i) {
        points_buffers_[i]->clear();
      }

    } else if (state_ != current_state) {
      state_ = current_state;

      // 処理ウィンドウの切替え
      std::swap(points_buffers_[0], points_buffers_[1]);
      std::swap(intensity_buffers_[0], intensity_buffers_[1]);
      points_buffers_[0]->clear();
      intensity_buffers_[0]->clear();
    }
    draw_mutex_.unlock();
  }


  void convertUrgData(PointsLine& line, double radian,
                      const long data[], int data_max_,
                      const Grid<int>& offset,
                      const long intensity_data[],
                      IntensityLine& intensity_line)
  {
    // ２次元展開
    std::vector<Grid<int> > points_2d;
    //convert2D(points_2d, &urg_, data, data_max_, &offset);
    convert2DWithIntensity(points_2d, intensity_line,
                           &urg_, data, data_max_,
                           intensity_data,
                           urg_min_distance_, urg_max_distance_, &offset);

    // ３次元変換
    for (std::vector<Grid<int> >::iterator it = points_2d.begin();
         it != points_2d.end(); ++it) {
      Grid3D<int> p(it->x, it->y, 0);
      rotateY(p, radian);

      line.push_back(p);
    }
  }

  void rotateY(Grid3D<int>& point, double radian)
  {
    double z = (point.z * cos(-radian)) - (point.x * sin(-radian));
    double x = (point.z * sin(-radian)) + (point.x * cos(-radian));

    point.z = static_cast<int>(z);
    point.x = static_cast<int>(x);
  }


  void redraw(void)
  {
    draw_mutex_.lock();
    for (int i = 0; i < DrawWindowsNum; ++i) {
      draw_widgets_[i].redraw(points_buffers_[i],
                              intensity_buffers_[i], (i == 0) ? true : false);
      draw_widgets_[i].updateCameraView();
    }
    draw_mutex_.unlock();
  }


  void initializePositionOrder(void)
  {
    const std::string motor_device =
      parent_->motor_combobox_->currentText().toStdString();
    motor_.connect(motor_device.c_str());
    motor_.initialize();
    motor_.disconnect();
  }
};


SwingScan3DWindow::SwingScan3DWindow(void)
  : QMainWindow(), pimpl(new pImpl(this))
{
  setupUi(this);
  pimpl->initializeForm();
}


SwingScan3DWindow::~SwingScan3DWindow(void)
{
}


void SwingScan3DWindow::rescan(void)
{
  pimpl->rescan();
}


void SwingScan3DWindow::urgSelected(void)
{
  pimpl->urgSelected();
}


void SwingScan3DWindow::motorSelected(void)
{
  pimpl->motorSelected();
}


void SwingScan3DWindow::connectSensors(void)
{
  pimpl->connectSensors();
}


void SwingScan3DWindow::disconnectSensors(void)
{
  pimpl->disconnectSensors();
}


void SwingScan3DWindow::capture(void)
{
  pimpl->capture();
}


void SwingScan3DWindow::redraw(void)
{
  pimpl->redraw();
}


void SwingScan3DWindow::config(void)
{
  pimpl->config_.show();
}


void SwingScan3DWindow::swingRange(int degree)
{
  pimpl->swing_thread_.setSwingRange(deg(degree));
}


void SwingScan3DWindow::swingOffset(int degree)
{
  pimpl->swing_offset_ = degree;
}


void SwingScan3DWindow::ignoreLength(int mm)
{
  pimpl->urg_min_distance_ = mm;
}


void SwingScan3DWindow::initializePositionOrder(void)
{
  pimpl->initializePositionOrder();
}


void SwingScan3DWindow::save(void)
{
  // 保存するファイル名の取得
  QString fileName =
    QFileDialog::getSaveFileName(this, tr("Save VRML file."), "./untitled.wrl",
                                 tr("VRML file (*.wrl)"));

  if (! fileName.isEmpty()) {
    // 日本語を含むパスへの対処
    std::string save_file = qrk::toStdStringPath(fileName);

    // 実際の保存処理は、DataDrawWidget で行う
    pimpl->draw_widgets_[1].save(save_file.c_str());
  }
}
