/*!
  \file
  \brief 操作ボタン付きの描画ウィジット

  \author Satofumi KAMIMURA

  $Id: ScanedControlWidget.cpp 246 2008-09-30 11:39:48Z satofumi $
*/

#include "ScanedControlWidget.h"
#include "DataDrawWidget.h"
#include "getTicks.h"
#include "LuaHandler.h"

using namespace qrk;

extern int ViewX;
extern int ViewY;
extern int ViewZ;
extern int Ticks;


namespace
{
  const static double MinRatio = 0.11;
  const static double MaxRatio = 10.00;
};


struct ScanedControlWidget::pImpl
{
  ScanedControlWidget* parent_;
  DataDrawWidget draw_widget_;

  int first_ticks_;
  bool is_auto_;


  pImpl(ScanedControlWidget* parent)
  : parent_(parent),
    first_ticks_(0), is_auto_(false)
  {
  }


  void initializeForm(void)
  {
    // 描画ウィジットの配置
    parent_->dummy_label_->hide();
    parent_->main_layout_->addWidget(&draw_widget_);

    // ズーム
    connect(parent_->zoom_slider_, SIGNAL(valueChanged(int)),
            parent_, SLOT(setZoomPercent(int)));

    // 視点変更
    connect(parent_->left_button_, SIGNAL(clicked()),
            parent_, SLOT(moveLeft()));
    connect(parent_->right_button_, SIGNAL(clicked()),
            parent_, SLOT(moveRight()));
    connect(parent_->up_button_, SIGNAL(clicked()),
            parent_, SLOT(moveUp()));
    connect(parent_->down_button_, SIGNAL(clicked()),
            parent_, SLOT(moveDown()));

    // カメラ視点の変更
    connect(parent_->auto_button_, SIGNAL(clicked(bool)),
            parent_, SLOT(autoButton(bool)));
  }


  void updateCameraView(void)
  {
    if (! is_auto_) {
      return;
    }

    Ticks = getTicks() - first_ticks_;
    LuaHandler* lua = LuaHandler::singleton();
    int error = lua->doFile("camera_control.lua");
    if (error != 0) {
      printf("Lua error %d\n", error);
    }
    Grid3D<int> rotation(ViewX, ViewY, ViewZ);
    draw_widget_.setRotation(rotation);
  }
};


ScanedControlWidget::ScanedControlWidget(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this))
{
  setupUi(this);

  pimpl->initializeForm();
}


ScanedControlWidget::~ScanedControlWidget(void)
{
}


PointsLine& ScanedControlWidget::emptyLine(void)
{
  return pimpl->draw_widget_.emptyLine();
}


void ScanedControlWidget::updateCameraView(void)
{
  pimpl->updateCameraView();
}


void ScanedControlWidget::redraw(Points3D* data, Intensity3D* intensity_data,
                                 bool isUpdating)
{
  pimpl->draw_widget_.redraw(data, intensity_data, isUpdating);
}


void ScanedControlWidget::setZoomPercent(int percent)
{
  zoom_slider_->setValue(percent);

  double ratio = MinRatio + (MaxRatio - MinRatio) * percent / 100.0;
  pimpl->draw_widget_.setZoomRatio(ratio);
}


int ScanedControlWidget::zoomPercent(void)
{
  return zoom_slider_->value();
}


void ScanedControlWidget::rotateLeft(void)
{
  pimpl->draw_widget_.rotateLeft();
}


void ScanedControlWidget::rotateRight(void)
{
  pimpl->draw_widget_.rotateRight();
}


void ScanedControlWidget::rotateUp(void)
{
  pimpl->draw_widget_.rotateUp();
}


void ScanedControlWidget::rotateDown(void)
{
  pimpl->draw_widget_.rotateDown();
}


void ScanedControlWidget::moveLeft(void)
{
  pimpl->draw_widget_.moveLeft();
}


void ScanedControlWidget::moveRight(void)
{
  pimpl->draw_widget_.moveRight();
}


void ScanedControlWidget::moveUp(void)
{
  pimpl->draw_widget_.moveUp();
}


void ScanedControlWidget::moveDown(void)
{
  pimpl->draw_widget_.moveDown();
}


void ScanedControlWidget::autoButton(bool checked)
{
  if (checked) {
    pimpl->first_ticks_ = getTicks();
  } else {
    pimpl->draw_widget_.setRotation(Grid3D<int>(0, 0, 0));
  }
  pimpl->is_auto_ = checked;
}


void ScanedControlWidget::save(const char* fname)
{
  pimpl->draw_widget_.save(fname);
}
