/*!
  \file
  \brief データ描画ウィジット

  \author Satofumi KAMIMURA

  $Id: DataDrawWidget.cpp 253 2008-10-02 06:42:39Z satofumi $
*/

#include "DataDrawWidget.h"
#include "vrmlPointHandler.h"
#include <QMouseEvent>

using namespace qrk;


struct DataDrawWidget::pImpl
{
  DataDrawWidget* parent_;
  Points3D* data_;
  Intensity3D* intensity_data_;
  bool draw_laser_;

  QColor clear_color_;
  Grid3D<int> position_;        //!< 描画グリッド位置
  Grid3D<int> rotation_;        //!< 描画視点
  QPoint last_position_;        //!< マウスのリリース位置
  double zoom_ratio_;           //!< ズーム比率

  bool updated_;


  pImpl(DataDrawWidget* parent)
    : parent_(parent), data_(NULL), draw_laser_(false),
      clear_color_(Qt::black), zoom_ratio_(1.0), updated_(false)
  {
  }


  void initializeGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glEnable(GL_TEXTURE_2D);
  }


  void paintGL(void)
  {
    parent_->qglClearColor(clear_color_);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    if (data_ == NULL) {
      return;
    }

    glLoadIdentity();

    glRotated(rotation_.x, 1.0, 0.0, 0.0);
    glRotated(rotation_.y, 0.0, 1.0, 0.0);
    glRotated(rotation_.z, 0.0, 0.0, 1.0);

    // 最新データのレーザ範囲を描画
    if (draw_laser_ && (! data_->empty())) {
      drawData(data_->front(), intensity_data_->front(),
               true, 0.0, 0.0, 0.4, 10);
    }

    // データの描画
    glBegin(GL_POINTS);
    Intensity3D::iterator intensity_it = intensity_data_->begin();
    for (Points3D::iterator it = data_->begin();
         it != data_->end(); ++it, ++intensity_it) {
      drawData(*it, *intensity_it, false, 1.0, 1.0, 1.0, 2);
    }
    glEnd();
  }


  // !!! 実装が汚い。作りなおすべき
  void drawData(const PointsLine& line_data,
                const IntensityLine& intensity_data,
                bool draw_laser, double r, double g, double b, int skip_line)
  {
    int index = 0;

    IntensityLine::const_iterator intensity_it = intensity_data.begin();
    PointsLine::const_iterator it_end = line_data.end();
    for (PointsLine::const_iterator it =
           line_data.begin(); it != it_end; ++it, ++intensity_it, ++index) {
      if ((index % skip_line) != 0) {
        continue;
      }

      if (! draw_laser) {
        double ratio = *intensity_it / 5000.0;
        glColor3d(r * ratio, g * ratio, b * ratio);
      } else {
        glColor3d(r, g, b);
      }

      if (draw_laser) {
        glBegin(GL_LINE_STRIP);
        glVertex3d(0.0, 0.0, 0.0);
      } else {
        glBegin(GL_POINTS);
      }
      glVertex3d(-(it->y - position_.x) * zoom_ratio_,
                 +(it->x - position_.y) * zoom_ratio_,
                 +(it->z - position_.z) * zoom_ratio_);
      glEnd();
    }
  }


  PointsLine& emptyLine(void)
  {
    PointsLine next_line;
    data_->push_front(next_line);
    return data_->front();
  }


  void mousePressEvent(QMouseEvent* event)
  {
    last_position_ = event->pos();
  }


  void resizeGL(int width, int height)
  {
    glViewport(0, 0, width, height);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    double aspect = 1.0 * width / height;
    enum { DefaultViewLength = 5000 };
    glOrtho(-DefaultViewLength * aspect, +DefaultViewLength * aspect,
            -DefaultViewLength, DefaultViewLength,
            std::numeric_limits<int>::min(), std::numeric_limits<int>::max());

    glMatrixMode(GL_MODELVIEW);
  }


  void mouseMoveEvent(QMouseEvent *event)
  {
    int dx = (event->x() - last_position_.x()) / 2;
    int dy = (event->y() - last_position_.y()) / 2;

    if (event->buttons() & Qt::LeftButton) {
      setXRotation(rotation_.x + dy);
      setYRotation(rotation_.y + dx);

    } else if (event->buttons() & Qt::RightButton) {
      setXRotation(rotation_.x + dy);
      setZRotation(rotation_.z - dx);
  }
    last_position_ = event->pos();
  }


  void setXRotation(int angle)
  {
    normalizeAngle(angle);
    if (angle != rotation_.x) {
      rotation_.x = angle;
    }
  }


  void setYRotation(int angle)
  {
    normalizeAngle(angle);
    if (angle != rotation_.y) {
      rotation_.y = angle;
    }
  }


  void setZRotation(int angle)
  {
    normalizeAngle(angle);
    if (angle != rotation_.z) {
      rotation_.z = angle;
    }
  }


  void normalizeAngle(int& angle)
  {
    while (angle < 0) {
      angle += 360;
    }

    while (angle > 360) {
      angle -= 360;
    }
  }


  void save(const char* fname)
  {
    // 保存データの作成
    std::vector<Grid3D<double> > points;
    std::vector<Color> colors;

    // !!! データ描画と共通化すべき
    int skip_line = 2;
    int line_index = 0;
    Intensity3D::iterator intensity_line_it = intensity_data_->begin();
    for (Points3D::iterator length_line_it = data_->begin();
         length_line_it != data_->end();
         ++length_line_it, ++intensity_line_it) {
      if ((line_index++ % skip_line) != 0) {
	continue;
      }

      int index = 0;

      IntensityLine::const_iterator intensity_it = intensity_line_it->begin();
      PointsLine::const_iterator it_end = length_line_it->end();
      for (PointsLine::const_iterator it = length_line_it->begin();
           it != it_end; ++it, ++intensity_it) {
        if ((index++ % skip_line) != 0) {
          continue;
        }

        double ratio = *intensity_it / 5000.0;
        Color point_color(ratio, ratio, ratio);
        colors.push_back(point_color);

        Grid3D<double> point_position(-it->y / 100.0,
				      it->x / 100.0,
				      it->z / 100.0);
        points.push_back(point_position);
      }
    }

    // 実際にファイル書き出し
    qrk::vrml_PointSave(fname, points, colors);
  }
};


DataDrawWidget::DataDrawWidget(QGLWidget* parent)
  : QGLWidget(parent), pimpl(new pImpl(this))
{
}


DataDrawWidget::~DataDrawWidget(void)
{
}


void DataDrawWidget::initializeGL(void)
{
  pimpl->initializeGL();
}


void DataDrawWidget::resizeGL(int width, int height)
{
  pimpl->resizeGL(width, height);
}


void DataDrawWidget::paintGL(void)
{
  pimpl->paintGL();
}


void DataDrawWidget::mousePressEvent(QMouseEvent *event)
{
  pimpl->mousePressEvent(event);
}


void DataDrawWidget::mouseMoveEvent(QMouseEvent *event)
{
  pimpl->mouseMoveEvent(event);
  pimpl->updated_ = true;
}


PointsLine& DataDrawWidget::emptyLine(void)
{
  return pimpl->emptyLine();
}


void DataDrawWidget::redraw(Points3D* data, Intensity3D* intensity_data,
                            bool isUpdating)
{
  pimpl->updated_ |= (pimpl->data_ != data) ? true : false;
  pimpl->data_ = data;
  pimpl->intensity_data_ = intensity_data;
  pimpl->draw_laser_ = isUpdating;
  if (pimpl->updated_ || isUpdating) {
    updateGL();
    pimpl->updated_ = false;
  }
}


void DataDrawWidget::setZoomRatio(const double ratio)
{
  pimpl->zoom_ratio_ = ratio;
  pimpl->updated_ = true;
}


void DataDrawWidget::zoomIn(void)
{
  pimpl->zoom_ratio_ *= 1.2;
  pimpl->updated_ = true;
}


void DataDrawWidget::zoomOut(void)
{
  pimpl->zoom_ratio_ *= 0.8;
  if (pimpl->zoom_ratio_ < 0.001) {
    pimpl->zoom_ratio_ = 0.001;
  }
  pimpl->updated_ = true;
}


double DataDrawWidget::zoomRatio(void)
{
  return pimpl->zoom_ratio_;
}


Grid3D<int> DataDrawWidget::rotation(void)
{
  return pimpl->rotation_;
}


void DataDrawWidget::setRotation(const qrk::Grid3D<int>& rotation)
{
  pimpl->rotation_ = rotation;
  pimpl->updated_ = true;
}


void DataDrawWidget::rotateLeft(void)
{
  pimpl->rotation_.y -= 4;
  pimpl->updated_ = true;
}


void DataDrawWidget::rotateRight(void)
{
  pimpl->rotation_.y += 4;
  pimpl->updated_ = true;
}


void DataDrawWidget::rotateUp(void)
{
  pimpl->rotation_.x -= 4;
  pimpl->updated_ = true;
}


void DataDrawWidget::rotateDown(void)
{
  pimpl->rotation_.x += 4;
  pimpl->updated_ = true;
}


Grid3D<int> DataDrawWidget::position(void)
{
  return pimpl->position_;
}


void DataDrawWidget::moveLeft(void)
{
  pimpl->position_.x += 100;
  pimpl->updated_ = true;
}


void DataDrawWidget::moveRight(void)
{
  pimpl->position_.x -= 100;
  pimpl->updated_ = true;
}


void DataDrawWidget::moveUp(void)
{
  pimpl->position_.y += 100;
  pimpl->updated_ = true;
}


void DataDrawWidget::moveDown(void)
{
  pimpl->position_.y -= 100;
  pimpl->updated_ = true;
}


void DataDrawWidget::save(const char* fname)
{
  pimpl->save(fname);
}
