/*!
  \file
  \brief 素振り回数の表示 Widget

  \aurthor Satofumi KAMIMURA

  $Id: SwingCounterWidget.cpp 301 2008-10-26 16:18:00Z satofumi $

  \todo Lua スクリプトを、バイナリの位置、フォルダの位置の順に探す
  \todo 接続終了時にも振動させる (２回の振動するとか)
  \todo 日付が変わったことを検出して、回数の再計算を行う
  \todo 設定フォームを、マウスドラッグで移動できるようにする
*/

#include "SwingCounterWidget.h"
#include "WiimoteConnectThread.h"
#include "SettingWidget.h"
#include "WiiJoystick.h"
#include "getTicks.h"
#include "LuaHandler.h"
#include "initializeLuaBind.h"
#include "split.h"
#include <QTimer>
#include <QDate>
#include <QMouseEvent>
#include <QShortcut>
#include <QSettings>
#include <map>
#include <cmath>
#include <fstream>

using namespace qrk;


namespace
{
  enum {
    ConnectSleepTime = 3 * 1000, // 再度接続を行うまでの待ち時間 [mse]
    DisconnectJudgeTime = 10 * 1000, // 加速度一定時の切断判定時間 [msec]

    CaptureIntervalTime = 10,   // 加速度の取得間隔 [msec]

    FirstRumbleTime = 1000,     // 接続直後の振動時間 [msec]
  };

  typedef std::map<std::string, int> SwingRecord;

  const double AccelerationError = 0.06;

  const char* Organization = "Hyakuren Soft";
  const char* Application = "SwingCounter";
};


struct SwingCounterWidget::pImpl
{
  SwingCounterWidget* parent_;
  QPoint drag_position_;
  QTimer capture_timer_;

  Grid3D<double> stopped_last_acc_;
  int stopped_first_;

  WiiJoystick wii_;
  LuaHandler* lua_handler_;
  lua_State* lua_;

  WiimoteConnectThread wii_connect_;

  int days_total_;
  int swing_times_;
  int today_times_;
  int average_times_;
  int total_times_;
  std::string log_file_;
  std::string today_string_;
  std::string data_directory_;

  SwingRecord swing_record_;

  SettingWidget setting_widget_;


  pImpl(SwingCounterWidget* parent)
  : parent_(parent), stopped_first_(0),
    lua_handler_(LuaHandler::singleton()), lua_(lua_handler_->luaPointer()),
    days_total_(1),
    swing_times_(0), today_times_(0), average_times_(0), total_times_(0),
    log_file_("swing_log.txt"), today_string_(createTodayString())
  {
  }


  void initializeForm(void)
  {
    connect(&wii_connect_, SIGNAL(wiimoteConnected()),
            parent_, SLOT(wiimoteConnected()));
    connect(&capture_timer_, SIGNAL(timeout()),
            parent_, SLOT(captureAccelerations()));
    connect(&setting_widget_, SIGNAL(logDirectoryChanged(const std::string&)),
            parent_, SLOT(logDirectoryChanged(const std::string&)));

    // 右クリックメニュー
    QAction* config_action = new QAction(QAction::tr("Config"), parent_);
    connect(config_action, SIGNAL(triggered()), parent_, SLOT(configHandler()));
    parent_->addAction(config_action);
    QAction* quit_action = new QAction(QAction::tr("E&xit"), parent_);
    quit_action->setShortcut(tr("Ctrl+Q"));
    connect(quit_action, SIGNAL(triggered()), parent_, SLOT(close()));
    parent_->addAction(quit_action);

    parent_->setContextMenuPolicy(Qt::ActionsContextMenu);
  }


  void saveSettings(void)
  {
    QSettings settings(Organization, Application);

    settings.setValue("geometry", parent_->saveGeometry());
    settings.setValue("data_directory", QString(data_directory_.c_str()));
  }


  void loadSettings(void)
  {
    QSettings settings(Organization, Application);

    parent_->restoreGeometry(settings.value("geometry").toByteArray());
    data_directory_ =
      settings.value("data_directory", "./").toString().toStdString();

    setting_widget_.setLogDirectory(data_directory_);
  }


  void connectWiimote(void)
  {
    wii_connect_.setWiimote(&wii_);
    wii_connect_.start();
  }


  void wiimoteConnected(void)
  {
    // 測定処理の開始
    stopped_first_ = getTicks();
    capture_timer_.start(CaptureIntervalTime);

    // 接続できたことを示すための振動
    parent_->rumble(FirstRumbleTime);
  }


  void disconnectWiimote(void)
  {
    capture_timer_.stop();
    wii_.disconnect();
    today_times_ += swing_times_;
    swing_times_ = 0;
    lua_clearSwingTimes(lua_);
    saveSwingLog();

    // 接続できるかの確認を開始
    // 即座に接続すると再接続してしまうので、長めに待っている
    waitAndConnect(ConnectSleepTime);
  }


  void captureAccelerations(void)
  {
    // 情報更新
    Grid3D<double> acc;
    wii_.getAcceleration(acc);
    int ticks = getTicks();

    // 素振り回数取得
    swing_times_ = lua_swingTimes(lua_, ticks, acc, parent_);
    updateAverageTimes();
    updateSwingTimes();

    if ((fabs(stopped_last_acc_.x - acc.x) < AccelerationError) &&
        (fabs(stopped_last_acc_.y - acc.y) < AccelerationError) &&
        (fabs(stopped_last_acc_.z - acc.z) < AccelerationError)) {

      // 加速度の変化なければ、素振りが終了したと判断し、Wii との接続を切断する
      if ((ticks - stopped_first_) > DisconnectJudgeTime) {
        disconnectWiimote();
      }

    } else {
      stopped_first_ = ticks;
      stopped_last_acc_ = acc;
    }
  }


  void waitAndConnect(int interval_msec)
  {
    QTimer::singleShot(interval_msec, parent_, SLOT(connectWiimote()));
  }


  bool loadSwingLog(void)
  {
    today_times_ = 0;
    average_times_ = 0;
    total_times_ = 0;

    // 素振り回数のログ読み出し
    std::string load_file = data_directory_ + "/" + log_file_;
    std::ifstream fin(load_file.c_str());
    if (! fin.is_open()) {
      // !!! ログファイルが開かれなかったメッセージを表示
      return false;
    }

    // !!! 以下、制御構造と処理がまざりすぎなので、分ける
    QDate today = QDate::currentDate();
    QDate oldest_date = today;
    size_t total_count = 0;
    size_t today_count = 0;
    size_t line_number = 0;
    std::string line;
    while (getline(fin, line)) {
      ++line_number;

      if (line.empty() || (line[0] == '#')) {
        // 先頭文字が '#' な場合はコメントとみなして無視
        continue;
      }

      std::vector<std::string> tokens;
      if (split(tokens, line, "\t ") != 2) {
        fprintf(stderr, "warnning: %d invalid data.\n", line_number);
        continue;
      }

      if (tokens[0][0] == '#') {
        // 最初のトークンの最初の文字が '#' な場合はコメントとみなして無視
        continue;
      }

      std::string date_string = tokens[0];
      QDate date;
      if ((! parseDate(date, date_string)) || (! date.isValid())) {
        continue;
      }

      if (oldest_date > date) {
        oldest_date = date;
      }
      int times = atoi(tokens[1].c_str());
      swing_record_[date_string] = times;

      if (date == today) {
        today_count = times;
        today_string_ = date_string;
      } else {
        total_count += times;
      }
    }

    days_total_ = oldest_date.daysTo(today) + 1;
    today_times_ = today_count;
    total_times_ = total_count;
    swing_times_ = 0;
    lua_clearSwingTimes(lua_);
    updateAverageTimes();
    updateSwingTimes();

    return true;
  }


  std::string createTodayString(void)
  {
    QDate today = QDate::currentDate();

    return today.toString("yyyy-MM-dd").toStdString();
  }


  bool parseDate(QDate& converted_date, const std::string& date_string)
  {
    std::vector<std::string> tokens;
    if (split(tokens, date_string, "-/") != 3) {
      return false;
    }

    converted_date = QDate(atoi(tokens[0].c_str()),
                           atoi(tokens[1].c_str()),
                           atoi(tokens[2].c_str()));
    return true;
  }


  void updateAverageTimes(void)
  {
    average_times_ = (total_times_ + today_times_ + swing_times_) / days_total_;
  }


  void updateSwingTimes(void)
  {
    parent_->today_counter_label_->
      setText(QString::number(today_times_ + swing_times_));
    parent_->average_counter_label_->setText(QString::number(average_times_));
    parent_->total_counter_label_->
      setText(QString::number(total_times_ + today_times_ + swing_times_));
  }


  bool saveSwingLog(void)
  {
    // !!! Windows において "//" が "/" に見なされないならば、修正すべき
    std::string save_file = data_directory_ + "/" + log_file_;
    std::ofstream fout(save_file.c_str());
    if (! fout.is_open()) {
      // !!! ファイルが開けない旨を出力すべき
      return false;
    }

    swing_record_[today_string_] = today_times_;

    for (SwingRecord::iterator it = swing_record_.begin();
         it != swing_record_.end(); ++it) {
      fout << it->first << '\t' << it->second << std::endl;
    }
    return true;
  }
};


SwingCounterWidget::SwingCounterWidget(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this))
{
  setupUi(this);
  pimpl->initializeForm();
  pimpl->loadSettings();
  initializeLuaBind(pimpl->lua_, pimpl->data_directory_.c_str());
  pimpl->loadSwingLog();

  // フォームの起動が終了するのを保証するため、しばらくしてから接続を開始する
  pimpl->waitAndConnect(ConnectSleepTime);
}


SwingCounterWidget::~SwingCounterWidget(void)
{
  pimpl->saveSwingLog();
  pimpl->saveSettings();
}


void SwingCounterWidget::mousePressEvent(QMouseEvent *event)
{
  if (event->button() == Qt::LeftButton) {
    pimpl->drag_position_ = event->globalPos() - frameGeometry().topLeft();
    event->accept();
  }
}


void SwingCounterWidget::mouseMoveEvent(QMouseEvent *event)
{
  if (event->buttons() & Qt::LeftButton) {
    move(event->globalPos() - pimpl->drag_position_);
    event->accept();
  }
}


void SwingCounterWidget::connectWiimote(void)
{
  pimpl->connectWiimote();
}


void SwingCounterWidget::captureAccelerations(void)
{
  pimpl->captureAccelerations();
}


void SwingCounterWidget::rumble(int rumble_msec)
{
  if (! pimpl->wii_.isConnected()) {
    return;
  }
  pimpl->wii_.setRumble(true);

  // 指定時間後に振動を停止させる
  QTimer::singleShot(rumble_msec, this, SLOT(stopRumble()));
}


void SwingCounterWidget::stopRumble(void)
{
  if (! pimpl->wii_.isConnected()) {
    return;
  }

  pimpl->wii_.setRumble(false);
}


void SwingCounterWidget::configHandler(void)
{
  pimpl->setting_widget_.show();
}


void SwingCounterWidget::wiimoteConnected(void)
{
  pimpl->wiimoteConnected();
}


void SwingCounterWidget::closeEvent(QCloseEvent* event)
{
  static_cast<void>(event);

  // 接続スレッドの終了
  pimpl->wii_connect_.terminate();
  pimpl->wii_connect_.wait();

  // 接続の切断とログ保存
  pimpl->disconnectWiimote();

  // 設定ウィンドウの消去
  pimpl->setting_widget_.close();
}


void SwingCounterWidget::logDirectoryChanged(const std::string& path)
{
  pimpl->data_directory_ = path;
}
