/*!
  \file
  \brief コマンド応答の記録

  \author Satofumi KAMIMURA

  $Id: LogRecordWidget.cpp 218 2008-09-12 22:24:26Z satofumi $
*/

#include "LogRecordWidget.h"
#include "SerialCtrl.h"
#include "ConnectionUtils.h"
#include "FindComPorts.h"
#include "RecordThread.h"
#include <QTimer>
#include <QSettings>
#include <QMessageBox>
#include <QDateTime>
#include <QShortcut>

using namespace qrk;

namespace
{
  const char* Organization = "Hokuyo LTD.";
  const char* Application = "MultiUrgViewer";

  enum {
    Timeout = 1000,
    ShortTimeout = 200,         // !!! エラーになるならば、ここを調整すること
  };
};


struct LogRecordWidget::pImpl
{
  LogRecordWidget* parent_;
  SerialCtrl con_;
  RecordThread* record_thread_;

  QTimer update_timer_;
  QTimer send_command_timer_;

  std::string send_command_;
  QDateTime first_time_;


  pImpl(LogRecordWidget* parent) : parent_(parent), record_thread_(NULL)
  {
  }


  void initializeForm(void)
  {
    // 接続
    parent_->com_combobox_->setSizeAdjustPolicy(QComboBox::AdjustToContents);
    connect(parent_->rescan_button_, SIGNAL(clicked()),
            parent_, SLOT(rescan()));
    connect(parent_->connect_button_, SIGNAL(clicked()),
            parent_, SLOT(connectCom()));
    connect(parent_->disconnect_button_, SIGNAL(clicked()),
            parent_, SLOT(disconnectCom()));
    setConnectEnable(true);
    rescan();

    // 表示の更新
    connect(&update_timer_, SIGNAL(timeout()), parent_, SLOT(updateCount()));

    // データ送信
    connect(&send_command_timer_, SIGNAL(timeout()),
            parent_, SLOT(sendCommand()));
    connect(parent_->interval_button_, SIGNAL(toggled(bool)),
            parent_, SLOT(toggledIntervalButton(bool)));

    setOnceCheck(true);
    toggledIntervalButton(false);

    // 終了
    (void) new QShortcut(Qt::CTRL + Qt::Key_Q, parent_, SLOT(close()));
    (void) new QShortcut(Qt::ALT + Qt::Key_F4, parent_, SLOT(close()));
  }


  void saveSettings(void)
  {
    QSettings settings(Organization, Application);
    settings.setValue("geometry", parent_->saveGeometry());

    // コマンド
    settings.setValue("command", parent_->command_edit_->text());
    settings.setValue("interval", parent_->interval_spinbox_->value());
    settings.setValue("is_once_checked", parent_->once_button_->isChecked());
  }


  void loadSettings(void)
  {
    QSettings settings(Organization, Application);

    // コマンド
    parent_->command_edit_->setText(settings.value("command", "").toString());
    int interval_msec = settings.value("interval", 1000).toInt();
    parent_->interval_spinbox_->setValue(interval_msec);
    bool isOnceChecked = settings.value("is_once_checked", true).toBool();
    setOnceCheck(isOnceChecked);
  }


  void setOnceCheck(bool check)
  {
    // !!! ラジオボタンなので、このメソッドは不要
    parent_->once_button_->setChecked(check);
    parent_->interval_button_->setChecked(! check);
  }


  void rescan(void)
  {
    // ポートの更新
    FindComPorts com_finder;
    com_finder.addBaseName("/dev/ttyACM");
    com_finder.addBaseName("/dev/usb/ttyUSB");
    std::vector<std::string> com_ports = com_finder.find();

    parent_->com_combobox_->clear();
    for (std::vector<std::string>::iterator it = com_ports.begin();
         it != com_ports.end(); ++it) {
      parent_->com_combobox_->addItem(it->c_str());
    }

    // ポートがなければ、接続させない
    parent_->connect_button_->setEnabled(! com_ports.empty());
  }


  // !!! 記録可能という名前に変更する
  void setConnectEnable(bool enable)
  {
    parent_->connect_button_->setEnabled(enable);
    parent_->disconnect_button_->setEnabled(! enable);
    parent_->com_combobox_->setEnabled(enable);
    parent_->fileName_edit_->setEnabled(enable);

    parent_->command_edit_->setEnabled(enable);
    parent_->once_button_->setEnabled(enable);
    parent_->interval_button_->setEnabled(enable);
    parent_->interval_spinbox_->setEnabled(enable);

    parent_->second_lcd_->setEnabled(! enable);
    parent_->count_lcd_->setEnabled(! enable);
  }


  void connectCom(void)
  {
    // !!! この接続処理は、時間がかかる可能性があるので、スレッド化する
    const std::string device =
      parent_->com_combobox_->currentText().toStdString();

    // !!! ボーレートを可変にする
    if (! con_.connect(device.c_str(), 115200)) {
      QMessageBox::warning(parent_, "Connection",
                           con_.what(), QMessageBox::Close);
      return;
    }
    setConnectEnable(false);

    // 初期化コマンドの送信
    initialize();

    // コマンドの送信開始
    send_command_ = captureCommand();
    if (parent_->once_button_->isChecked()) {
      sendCommand();
    } else {
      int interval_msec = parent_->interval_spinbox_->value();
      send_command_timer_.start(interval_msec);
    }

    // 記録の開始
    std::string file_name = parent_->fileName_edit_->text().toStdString();
    record_thread_ = new RecordThread(&con_, file_name.c_str());
    record_thread_->start();

    update_timer_.start(1000);

    first_time_ = QDateTime::currentDateTime();
  }


  void initialize(void)
  {
    // !!! スレッド内で処理するようにする
    // !!! Lua スクリプトで調整できるようにする

    con_.send("QT\r", 3);
    skip(&con_, ShortTimeout);
    con_.send("SCIP2.0\r", 8);
    skip(&con_, ShortTimeout);
    con_.send("BM\r", 3);
    skip(&con_, ShortTimeout);
  }


  std::string captureCommand(void)
  {
    std::string command = parent_->command_edit_->text().toStdString();
    command.push_back('\r');

    return command;
  }


  void sendCommand(void)
  {
    con_.send(send_command_.c_str(), send_command_.size());
  }


  // !!! このメソッドを、close() で呼び出すべき
  void disconnectCom(void)
  {
    send_command_timer_.stop();
    update_timer_.stop();
    record_thread_->stop();
    record_thread_->wait();
    delete record_thread_;
    record_thread_ = NULL;

    con_.disconnect();
    setConnectEnable(true);
  }


  void updateCount(void)
  {
    parent_->count_lcd_->display(record_thread_->count());
    int recording_second = first_time_.secsTo(QDateTime::currentDateTime());
    parent_->second_lcd_->display(recording_second);
  }


  void toggledIntervalButton(bool checked)
  {
    parent_->interval_spinbox_->setEnabled(checked);
  }
};


LogRecordWidget::LogRecordWidget(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this))
{
  setupUi(this);
  pimpl->initializeForm();
  pimpl->loadSettings();
}


LogRecordWidget::~LogRecordWidget(void)
{
  pimpl->saveSettings();
}


void LogRecordWidget::rescan(void)
{
  pimpl->rescan();
}


void LogRecordWidget::connectCom(void)
{
  pimpl->connectCom();
}


void LogRecordWidget::disconnectCom(void)
{
  pimpl->disconnectCom();
}


void LogRecordWidget::updateCount(void)
{
  pimpl->updateCount();
}


void LogRecordWidget::sendCommand(void)
{
  pimpl->sendCommand();
}


void LogRecordWidget::toggledIntervalButton(bool checked)
{
  pimpl->toggledIntervalButton(checked);
}


void LogRecordWidget::initialize(void)
{
  pimpl->initialize();
}
