/*!
  \file
  \brief 視点の操作ウィジット

  \author Satofumi KAMIMURA

  $Id: ViewControlWidget.cpp 199 2008-09-02 10:55:59Z satofumi $

  \todo センサ中心を下から 1/3 の位置に修正する
  \todo 接続処理をスレッド化して、全体が停止しないようにする
  \todo 接続中は、ポート名のコンボボックスを disable にする
*/

#include "ViewControlWidget.h"
#include "RangeDataDrawWidget.h"
#include "UrgCtrl.h"
#include "AngleTypes.h"
#include "convert2D.h"
#include <QMutex>
#include <QTimer>

using namespace qrk;


struct ViewControlWidget::pImpl
{
  ViewControlWidget* parent_;
  RangeDataDrawWidget draw_widget_;

  UrgCtrl urg_;
  QMutex mutex_;
  QTimer rescan_timer_;

  long* data_;
  int data_max_;
  int scan_msec_;
  long max_distance_;


  pImpl(ViewControlWidget* parent)
    : parent_(parent), data_(NULL), data_max_(0), scan_msec_(100),
      max_distance_(30000)
  {
  }


  void initializeForm(void)
  {
    // 描画ウィジットの配置
    parent_->main_layout_->addWidget(&draw_widget_);
    parent_->dummy_label_->hide();

    // 接続
    parent_->port_combobox_->setSizeAdjustPolicy(QComboBox::AdjustToContents);
    connect(parent_->connect_button_, SIGNAL(clicked()),
            parent_, SLOT(connectUrg()));
    connect(parent_->disconnect_button_, SIGNAL(clicked()),
            parent_, SLOT(disconnectUrg()));
    setConnectEnable(true);

    // ズーム操作
    connect(parent_->zoom_slider_, SIGNAL(valueChanged(int)),
            parent_, SLOT(zoomChanged(int)));
    int initial_percent = 20;
    parent_->zoom_slider_->setValue(initial_percent);
    parent_->zoomChanged(initial_percent);

    // データ描画
    connect(&rescan_timer_, SIGNAL(timeout()), parent_, SLOT(capture()));
  }


  void connectUrg(void)
  {
    // !!! この接続処理は、時間がかかる可能性があるので、スレッド化する
    const std::string device =
      parent_->port_combobox_->currentText().toStdString();
    if (! urg_.connect(device.c_str())) {
      // !!! MessageBox の表示
      return;
    }
    updateUrgSettings();
    setConnectEnable(false);

    rescan_timer_.start(scan_msec_);
  }


  void disconnectUrg(void)
  {
    rescan_timer_.stop();
    urg_.disconnect();
    setConnectEnable(true);
  }


  void updateUrgSettings(void)
  {
    int urg_data_max = urg_.maxScanLines();
    if (urg_data_max > data_max_) {
      data_max_ = urg_data_max;
      delete [] data_;
      data_ = new long [data_max_];
    }
    scan_msec_ = urg_.scanMsec();
    max_distance_ = urg_.maxDistance();

    urg_.setCaptureMode(AutoCapture);
  }


  void setConnectEnable(bool enable)
  {
    parent_->connect_button_->setEnabled(enable);
    parent_->disconnect_button_->setEnabled(! enable);
  }


  void capture(void)
  {
    // データ取得
    // !!! 可能ならば、draw_widget_ から書き込み先バッファを受け取るようにする
    // !!! そうすることで、データのコピーを減らせる
    int n = urg_.capture(data_, data_max_);
    if (n <= 0) {
      return;
    }

    // 描画ウィジットへのデータ登録
    std::vector<Grid<int> > points;
    Position<int> offset(0, 0, deg(90));
    convert2D(points, &urg_, data_, n, &offset);

    mutex_.lock();
    draw_widget_.setData(points);
    mutex_.unlock();
  }


  void zoomChanged(int percent)
  {
    QSize widget_size = draw_widget_.size();
    const int w = widget_size.width();
    const int h = widget_size.height();
    int min_length = (w > h) ? h : w;

    // 0 % のときに、測定最大距離が画面に収まるようにする
    // !!! 厳密に実現するならば、resize を検出して拡大率を調整すべき
    double min_zoom = min_length / (max_distance_ * 2.0);
    double max_zoom = 1.0;
    double zoom = min_zoom + (max_zoom - min_zoom) * percent / 100.0;
    draw_widget_.setZoom(zoom);
  }
};


ViewControlWidget::ViewControlWidget(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this))
{
  setupUi(this);
  pimpl->initializeForm();
}


ViewControlWidget::~ViewControlWidget(void)
{
}


void ViewControlWidget::updatePorts(const std::vector<std::string>& ports,
                                    int defaultIndex)
{
  port_combobox_->clear();
  for (std::vector<std::string>::const_iterator it = ports.begin();
       it != ports.end(); ++it) {
    port_combobox_->addItem(it->c_str());
  }
  port_combobox_->setCurrentIndex(defaultIndex);

  // 現在のポート名が空ならば、接続させない
  if (0) {
    pimpl->setConnectEnable(false);
  }
}


void ViewControlWidget::connectUrg(void)
{
  pimpl->connectUrg();
}


void ViewControlWidget::disconnectUrg(void)
{
  pimpl->disconnectUrg();
}


void ViewControlWidget::capture(void)
{
  pimpl->capture();
}


void ViewControlWidget::redraw(void)
{
  pimpl->mutex_.lock();
  pimpl->draw_widget_.redraw();
  pimpl->mutex_.unlock();
}


bool ViewControlWidget::isConnected(void)
{
  // !!! ここでの urg_ 操作の排他保証は、UrgCtrl で実現されるべきなので、
  // !!! このプログラムでは対処しない
  // !!! ただし、現状では UrgCtrl におけるメソッド間での排他は実装されていない
  return pimpl->urg_.isConnected();
}


void ViewControlWidget::zoomChanged(int value)
{
  pimpl->zoomChanged(value);
}


int ViewControlWidget::zoomPercent(void)
{
  return zoom_slider_->value();
}


void ViewControlWidget::setZoomPercent(int percent)
{
  zoom_slider_->setValue(percent);
}
