/*!
  \file
  \brief 複数 URG の距離データ表示アプリケーション

  \author Satofumi KAMIMURA

  $Id: MultiUrgViewerWindow.cpp 197 2008-09-02 04:54:13Z satofumi $
*/

#include "MultiUrgViewerWindow.h"
#include "ViewControlWidget.h"
#include "FindComPorts.h"
#include "UrgUsbCom.h"
#include <QSettings>
#include <QTimer>

using namespace qrk;


namespace
{
  const char* Organization = "Hokuyo LTD.";
  const char* Application = "MultiUrgViewer";

  enum {
    DefaultZoomPercent = 20,
  };
};


struct MultiUrgViewerWindow::pImpl
{
  enum {
    UrgWidgetsTotal = 4,
    RedrawMsec = 100,
  };

  MultiUrgViewerWindow* parent_;
  ViewControlWidget control_widget_[UrgWidgetsTotal];

  QTimer redraw_timer_;


  pImpl(MultiUrgViewerWindow* parent) : parent_(parent)
  {
  }


  void initializeForm(void)
  {
    // 描画ウィジットの配置
    parent_->main_layout_->addWidget(&control_widget_[0], 0, 0);
    parent_->main_layout_->addWidget(&control_widget_[1], 0, 1);
    parent_->main_layout_->addWidget(&control_widget_[2], 1, 0);
    parent_->main_layout_->addWidget(&control_widget_[3], 1, 1);

    // メニュー
    connect(parent_->action_quit_, SIGNAL(triggered()), parent_, SLOT(close()));

    // 接続
    connect(parent_->rescan_button_, SIGNAL(clicked()),
            parent_, SLOT(rescan()));
    rescan();

    // 再描画の開始
    for (int i = 0; i < UrgWidgetsTotal; ++i) {
      connect(&redraw_timer_, SIGNAL(timeout()),
              &control_widget_[i], SLOT(redraw()));
    }
    redraw_timer_.start(RedrawMsec);
  }


  void saveSettings(void)
  {
    QSettings settings(Organization, Application);
    settings.setValue("geometry", parent_->saveGeometry());

    // 拡大率
    for (int i = 0; i < UrgWidgetsTotal; ++i) {
      std::string tag = zoomPercentString(i);
      settings.setValue(tag.c_str(), control_widget_[i].zoomPercent());
    }
  }


  void loadSettings(void)
  {
    QSettings settings(Organization, Application);
    parent_->restoreGeometry(settings.value("geometry").toByteArray());
    // !!! resize() イベントを適切に呼び、同じ位置に配置されるようにする

    // 拡大率
    for (int i = 0; i < UrgWidgetsTotal; ++i) {
      std::string tag = zoomPercentString(i);
      int percent = settings.value(tag.c_str(), DefaultZoomPercent).toInt();
      control_widget_[i].setZoomPercent(percent);
    }
  }


  std::string zoomPercentString(int index)
  {
    char buffer[] = "zoom_percent/xx";
    snprintf(buffer, strlen(buffer), "zoom_percent/%d", index);

    return buffer;
  }


  void rescan(void)
  {
    UrgUsbCom urg_usb;
    FindComPorts urg_finder(&urg_usb);
    std::vector<std::string> urg_ports = urg_finder.find();

    // 接続済みのポートがあれば、それは他のポートには追加しない
    // !!! control_widget_[] をループでまわし、接続済みポート名を取得する
    // !!! 接続中は、ポートのコンボボックスを disable にしてしまう

    for (int i = 0; i < UrgWidgetsTotal; ++i) {
      control_widget_[i].updatePorts(urg_ports, i);
    }
  }
};


MultiUrgViewerWindow::MultiUrgViewerWindow(void)
  : QMainWindow(), pimpl(new pImpl(this))
{
  setupUi(this);
  pimpl->initializeForm();
  pimpl->loadSettings();
}


MultiUrgViewerWindow::~MultiUrgViewerWindow(void)
{
  pimpl->saveSettings();
}


void MultiUrgViewerWindow::rescan(void)
{
  pimpl->rescan();
}
