#ifndef CDROM_PLAY_H
#define CDROM_PLAY_H

/*!
  \file
  \brief CD-ROM の再生

  \author Satofumi KAMIMURA

  $Id: CdromPlay.h 167 2008-08-24 01:01:01Z satofumi $

  \todo CD-ROM の有無を調べるだけのメソッド追加を検討する
*/

#include <memory>
#include <vector>


namespace qrk
{
  /*!
    \brief CD-ROM まわりの制御クラス
  */
  class CdromPlay
  {
    CdromPlay(const CdromPlay& rhs);
    CdromPlay& operator = (const CdromPlay& rhs);

    struct pImpl;
    const std::auto_ptr<pImpl> pimpl;

  public:

    /*!
      \brief トラックのデータタイプ
    */
    typedef enum {
      AudioTrack,               //!< 音楽トラック
      DataTrack,                //!< データトラック
    } TrackType;

    /*!
      \brief トラック情報
    */
    typedef struct {
      int id;                   //!< トラック番号
      TrackType type;           //!< データタイプ
      int frame_length;         //!< フレーム長
      int min;                  //!< 分
      int sec;                  //!< 秒
    } track_t;

    CdromPlay(void);


    /*!
      \brief 使用ドライブ指定のコンストラクタ
    */
    explicit CdromPlay(size_t drive_id);
    ~CdromPlay(void);


    /*!
      \brief ドライブ数の取得

      \retrun CD-ROM ドライブ数
    */
    static size_t drives(void);


    /*!
      \brief 状態の取得

      \return 状態を示す文字列
    */
    const char* what(void);


    /*!
      \brief トラック情報の取得

      \param[out] tracks トラック情報の格納先

      \retval true 正常終了
      \retval false エラー
    */
    bool tracks(std::vector<track_t>& tracks);


    /*!
      \brief 再生

      再生長さを指定する frame は １秒あたり 75 の単位系である。

      \param[in] track_id 再生するトラック ID
      \param[in] start_frame 再生開始オフセット位置 [frame]
      \param[in] play_frame 再生する長さ [frame]
      \param[in] tracks 再生するトラック数
    */
    int playTrack(size_t track_id,
                  size_t start_frame = 0,
                  size_t play_frame = 0,
                  size_t tracks = 1);


    /*!
      \brief 再生中かの取得

      \retval true 再生中
      \retval false それ以外

      \bug 短い周期で呼び出しを繰り返すと再生がぶれる。SDL-1.2 の仕様か？
    */
    bool isPlaying(void);


    /*!
      \brief 再生の停止
    */
    void stop(void);


    /*!
      \brief 再生の一時停止
    */
    void pause(void);


    /*!
      \brief 再生の再開
    */
    void resume(void);


    /*!
      \brief CD-ROM の取り出し
    */
    void eject(void);
  };
};

#endif /* !CDROM_PLAY_H */
