/*!
  \file
  \brief CD-ROM の再生

  \author Satofumi KAMIMURA

  $Id: CdromPlay.cpp 187 2008-08-30 10:26:11Z satofumi $
*/

#include <string>
#include <SDL.h>
#include "CdromPlay.h"
#include "SdlCdromInit.h"
#include "VolumeChangeInterface.h"
#include "delay.h"

using namespace qrk;


struct CdromPlay::pImpl : private SdlCdromInit, public VolumeChangeInterface
{
  class CdromInit : private SdlCdromInit
  {
  };

  size_t drive_id_;
  std::string error_message_;
  SDL_CD* cdrom_;
  std::vector<track_t> track_list_;
  CDstatus cd_status_;


  pImpl(size_t id)
    : drive_id_(id), error_message_("no error"), cdrom_(NULL),
      cd_status_(CD_ERROR)
  {
    openDrive(drive_id_);
  }


  ~pImpl(void)
  {
    if (cdrom_) {
      stop();
      SDL_CDClose(cdrom_);
    }
  }


  // CD-ROM ドライブの初期化
  bool openDrive(size_t id)
  {
    if (SDL_CDNumDrives() <= 0) {
      error_message_ = "not found CD-ROM drive.";
      return false;
    }

    cdrom_ = SDL_CDOpen(0);
    if (! cdrom_) {
      error_message_ = SDL_GetError();
      return false;
    }
    return true;
  }


  // ドライブのチェックと情報取得
  bool checkDrive(std::vector<track_t>& tracks)
  {
    tracks.clear();
    if (! cdrom_) {
      if (! openDrive(drive_id_)) {
        return false;
      }
    }

    cd_status_ = SDL_CDStatus(NULL);
    if (! CD_INDRIVE(cd_status_)) {
      error_message_ = "not found CD-ROM.";
      return false;
    }

    for (int i = 0; i < cdrom_->numtracks; ++i) {

      track_t info;
      SDL_CDtrack& track = cdrom_->track[i];
      info.id = track.id;
      info.frame_length = track.length;
      info.type = (track.type == SDL_AUDIO_TRACK) ? AudioTrack : DataTrack;

      int frame;
      FRAMES_TO_MSF(track.length, &info.min, &info.sec, &frame);

      tracks.push_back(info);
    }

    return true;
  }


  bool play(size_t id, size_t start_frame, size_t play_frame)
  {
    if (! checkDrive(track_list_)) {
      return false;
    }

    if ((id >= track_list_.size()) || (track_list_[id].type == DataTrack)) {
      return false;
    }

    int ntrack = (play_frame == 0) ? 1 : 0;
    SDL_CDPlayTracks(NULL, static_cast<int>(id),
                     static_cast<int>(start_frame),
                     ntrack, static_cast<int>(play_frame));
    return true;
  }


  void stop(void)
  {
    SDL_CDStop(NULL);
  }


  bool getCurrentFrame(size_t* frame, size_t* min, size_t* sec)
  {
#if 0
    if (! checkDrive(track_list_)) {
      return false;
    }
#endif

    // 値の取得
    size_t frame_length = 0;
    size_t min_length = 0;
    size_t sec_length = 0;
    FRAMES_TO_MSF(cdrom_->cur_frame, &min_length, &sec_length, &frame_length);
    fprintf(stderr, "%d, %d, %d\n", frame_length, min_length, sec_length);

    // 代入
    if (frame) {
      *frame = frame_length;
    }
    if (min) {
      *min = min_length;
    }
    if (sec) {
      *sec = sec_length;
    }
    return true;
  }
};


CdromPlay::CdromPlay(void) : pimpl(new pImpl(0))
{
}


CdromPlay::CdromPlay(size_t drive_id) : pimpl(new pImpl(drive_id))
{
}


CdromPlay::~CdromPlay(void)
{
}


size_t CdromPlay::drives(void)
{
  // CD-ROM システムの初期化を行わせる
  static pImpl::CdromInit audio_init;

  return SDL_CDNumDrives();
}


const char* CdromPlay::what(void)
{
  return pimpl->error_message_.c_str();
}


int CdromPlay::playTrack(size_t track_id,
                         size_t start_frame, size_t play_frame,
                         size_t tracks)
{
  if (! pimpl->play(track_id, start_frame, play_frame)) {
    return -1;
  }

  size_t play_frame_length = (play_frame == 0) ?
    pimpl->track_list_[track_id].frame_length - start_frame : play_frame;

  size_t play_msec = 1000 * play_frame_length / CD_FPS;
  return play_msec;
}


bool CdromPlay::tracks(std::vector<track_t>& tracks)
{
  return pimpl->checkDrive(tracks);
}


bool CdromPlay::isPlaying(void)
{
  pimpl->checkDrive(pimpl->track_list_);
  return (pimpl->cd_status_ == CD_PLAYING) ? true : false;
}


void CdromPlay::stop(void)
{
  pimpl->stop();
}


void CdromPlay::pause(void)
{
  SDL_CDPause(NULL);
}


void CdromPlay::resume(void)
{
  SDL_CDResume(NULL);
}


void CdromPlay::eject(void)
{
  SDL_CDEject(NULL);
}
