/*!
  \file
  \brief ログデータの 2D 変換サンプル

  \author Satofumi KAMIMURA

  $Id: logConvert2D.cpp 136 2008-07-29 10:27:20Z satofumi $
*/

#include <cmath>
#include "ScipHandler.h"
#include "SensorParameter.h"
#include "UrgCtrl.h"

using namespace qrk;


//! main
int main(int argc, char *argv[]) {

  const char gd_lines[][80] = {
#if 0
    "GD0000108000",
    "00P",
    "5eZIm",
    "0n^0ne0nm0o=0oY0oh10:10>10K10S10o10o10o0Yh0Yb0Yb0Yb0Y^0Y^0Y[0YJ0F",
    "Y90Y70Y70Y40Xj0Xi0X`0XJ0XH0XH0XH0XH0XB0X30Wo0Wb0Wb0Wb0Wd0Wl0X80X8",
    "H0X\\0Xm0Y10YK0YT0Yb0Z60ZD0ZD0ZD0ZC0SH0Rc0MO0Lh0LM0L40K[0KT0KO0KO0",
    "0K40J^0JX0JX0JS0JQ0JQ0JH0J?0J?0J:0J90J30In0In0Il0Ig0Ig0IY0IV0IV0Y",
    "IV0IV0IV0IQ0IQ0IQ0IZ0IZ0I\\0Ie0If0J[0L20La0La0La0L]0L]0L]0Lb0Ld0Lb",
    "d0Ld0Lb0Ld0Li0Lj0M10MI0Ma0Mm0Mn0N00N00Mm0Ml0Ml0Ml0N10N:0N]0OG0PUB",
    "0U[0X90YG0YR0YU0YY0YY0YY0YW0YW0YU0YQ0YF0Wf0Wf0Wf0Y10Y10Y10VF0US0P",
    "UJ0Tn0Tc0TZ0TY0TR0TR0TR0T^0UT0V10V10V10UY0UQ0UQ0U50U20U20U20UH0U@",
    "H0UH0RW0QK0Pl0Pd0Pd0PZ0PJ0PJ0PJ0P\\0Po0Q:0QR0QX0Qc0RH0RT0RT0RT0R82",
    "0Pc0Pc0Pc0Pc0Q>0RL0T70TD0TK0TU0U30Ue0V10V10V10PA0N^0N90Mn0Mk0Mi08",
    "MT0MR0MR0MR0MS0MT0MT0MT0MP0MP0MP0MP0MP0MW0MW0MW0MW0MZ0MZ0MZ0MZ0Mk",
    "Z0MY0MY0MY0MT0MJ0M80M80M80MF0MF0MF0M?0M?0M?0MA0MC0OP0SQ0SQ0SQ0SMY",
    "0SM0SM0SY0SY0S[0S[0S\\0S]0T80TB0Tf0X[0Zi0Zi0Zi0Z70Y30WW0U60U60U60g",
    "VF0VF0VF0VF0VL0VM0VT0V]0Vj0Vl0WE0WE0WE0WD0WD0WD0W;0W;0W90VY0Ul0UR",
    "_0U_0U_0Um0VR0VR0VR0Ue0U90Ro0RB0RB0RB0SH0Uh0Wc0Y_0ZO0ZP0Zo0[^0^dG",
    "0a10aG0aI0aL0aL0aL0`A0^`0^`0^`0_90_90_90_90_90a<0b00bL0bV0be0b^0j",
    "b^0b^0a70a70`X0`W0YI0Tg0T=0Sd0Sc0SX0SX0SX0Sd0Sd0Sg0Sl0Sm0T90TH0T?",
    "I0TJ0TR0TX0T\\0Tj0U60U@0UD0UN0UU0V30VE0VY0W00X<0ZD0a=0b>0bE0bJ0bg3",
    "0c?0d80eH0k40nD0nX0o70oN0oO0of10;10>10N10O11211V11n12612H12k13J1o",
    "3J13J13<0o50lU0lO0kR0kG0k60k60k60k30j80ib0i^0iW0iL0iL0iL0iY0i]0j`",
    "A0jQ0ja0k20k70k]0l00lS0lS0lS0:109Y09W09G09<09309108n08j08Z08O08=5",
    "07l06`05U05404l04l04l04h04h04g04g04f04g04g04f04f04k04k04k04k04m0k",
    "4i04i04h04j04h04f04d04f04c04f04g04g04c04]04]04]04g04g04g04a04a04\\",
    "a04a04a04^04^04^04_04b04b04b04^04^04^04f04f04f04f04f04c04c04c04cJ",
    "04c04b04b04b04`04[04Y04Y04[04[04Y04W04W04W04]04`04`04`04_04_04_08",
    "4g04g04g04a04a04a04a04b04b04b04b04b04b04c04c04a04a04a04_04_04_04c",
    "`04`04a04a04a04e04e04e04e04e04e04e04e04e04e04f04m05005004m04h04eh",
    "04e04d04d04e04e04e04i04i04k04k04h04h04g04g0520530540560540540520f",
    "4n04n04n05205205204o04n04n04o04n05105B0:60;J0<J0<J0<J0<Y0>=0>e0><",
    "e0>e0>40>00=E0=D0=@0<^0;g0;f0;_0;_0;X0;X0;X0:l09`09Q09C08f08M08M?",
    "08M08`09609W0;<0>\\0@30@30@70@>0@N0@R0@W0@c0@c0@c0@d0@i0@i0@n0A00M",
    "A40A<0A<0A<0A<0A:0A:0A:0A?0AE0AE0AL0AL0AX0AX0AX0AP0AK0@o0?U0>i0>0",
    "?0>90>70=k0=Y0=Y0=W0=W0=S0=J0<L0;a0;U0;90;80;70;00:k0:f0:f0:`0:`o",
    "0:f0:i0:i0:l0:l0:i0:l0:l0;00;B0;B0;B0;80:\\0:V0:U0:S0:K0:K0:K0:R0c",
    ":V0:]0:b0:d0;80;d0=10A00A00A00>W0=N0=>0<d0<d0<d0<k0=M0?i0A00A20AM",
    "20A20A20A20A30A40B>0C30CU0CU0CU0CU0CU0Ca0D10D10D10@b0@60@60?h0?g`",
    "0?g0?a0?_0?[0>m0>V0>V0>V0?D0@90@G0@G0@G0@?0@?0@<0@<0@90?e0?`0?R0I",
    "?R0?R0?R0?`0@E0AD0Ca0D[0D[0D[0C`0C`0C_0CN0C<0Ba0Ba0Ba0C90EJ0FL0FA",
    "L0F\\0Fk0G40H40H40H40GC0F[0FK0FC0E`0Dc0C;0@k0@f0@f0@f0@f0@f0@_0@V[",
    "0@R0@R0@M0@M0@M0@M0@M0@N0@N0@^0BD0C90CM0CQ0CT0CT0CT0@S0;40::09T0]",
    "9T09T09T09X09Y09l0:C0;U0>J0Ai0B40BA0BI0BR0Bl0C40CO0CO0CO0Bn0Ba0BV",
    "00AF0@e0@J0?S02001h01`01^01^01V01V01V01R01R01R01Q01K01K01Q01Y01Y0",
    "01Y01Y01Y01R01R01S01R01E01E01E01E01A01A01A01A01A01A01C01A01A01?0G",
    "1601601501501601;01;01;01<01L01L01L01L01L01L01M01M01R01T01R01T01=",
    "Y01T01R01M01K01K01K01K01K01E01C01C01?01?01?01Q01S01S01S01K01K01IY",
    "01J01J01I01E01E01E01M01N01N01N01N01N01I01U01U01I01:01801801:01B06",
    "1E01E01E01E01E01H01K01K01K01K01K01M01M01M01M01M01M01N01U01U01U01I",
    "U01U01^01^01^01O01O01O01S01[01[01[01[01[01g01g01g01f01Y01Y01\\01\\5",
    "01\\01^01\\01^01^02602:03M04B04P05605M05Q05Q05Q05O05O05O05N05R05V0E",
    "5R05R05R05H05H05H05C05A05A05=05905805605605305205205205104b04b04V",
    "b04b04`04`04^04^04Y04Y04W04W04W04X04X04X04X?"

#elif 0
    "GD0000108000",
    "00P",
    "3[Y<S",
    "0n\\0nh0o;0oP0o^0od10310=10Z10e11<11<11<0Yb0Yb0Yb0Yo0Yo0Yo0Y`0YB0S",
    "YB0Y60Y30Y20Y00Xb0X^0XU0XS0XP0XP0X:0X:0X50X30X20Wn0Wn0Wn0Wn0X=0XI",
    "N0Xe0Xg0Y10YE0YY0Yl0Ym0ZC0ZS0Zd0[90[D0[U0[X0\\90\\E0\\N0\\N0\\N0Z60X[d",
    "0X@0X@0X@0XU0XU0XU0U00RE0P[0O>0Nd0NK0NH0NE0N90Mo0M]0MZ0MR0MP0MO0S",
    "MN0MC0M10M10Ln0Ln0Ln0Lj0Lb0Lb0Lb0Lb0Lb0L]0L]0L]0L]0L]0L]0LX0LX0L[",
    "X0LX0LZ0Ld0Ld0Lg0Lh0Ll0Lo0M40M60M<0M=0ME0MP0MV0N40N60NB0NE0Na0ONZ",
    "0Q80Y00YM0YN0YP0YR0YV0YV0YV0YD0Y;0Y40Xe0TT0R>0Qb0Q]0Qb0Qh0Qh0R:0U",
    "RO0Re0SF0Sn0TJ0TK0TS0TY0TZ0UP0Ui0Ui0Ui0UW0UA0U50U30Tk0Tk0Tk0UL0U7",
    "L0UL0Rh0Q@0Pj0PX0PT0PT0PL0PL0PL0P]0Pd0Q60QJ0QT0Q\\0RK0RZ0RZ0RZ0R5S",
    "0P`0P`0P`0Ph0Q50RL0Sg0T@0TO0TP0Ti0V10V=0V=0V=0P<0Nl0N80Mi0Mi0Mi0<",
    "MX0MT0MS0MS0MS0MS0MN0MM0MM0MM0MS0MS0MS0MS0MS0MV0MV0MV0MR0MR0MR0MB",
    "T0MT0MX0MT0ML0MH0M<0M<0M<0M<0MA0MA0MA0M=0M70M70M70Ol0Sb0Sb0Sb0SPj",
    "0SP0SP0SX0SX0SZ0S[0S`0Sd0Se0T:0Tf0Y50ZV0ZV0ZV0Yh0Y80Wa0Te0Te0Te0D",
    "V?0VB0VB0VC0VC0VM0VN0VU0Vo0W50W80W90W;0W<0WA0WA0WA0W60Vk0Vd0VB0U[",
    "j0UR0UR0UR0Uf0V00V00V00U50Rn0RM0RM0RM0SA0Ub0WJ0Ya0ZV0ZZ0Zg0[_0^]3",
    "0a;0aC0aL0aR0aR0aR0`I0_40_40_40^b0^b0^b0^b0^b0a>0b40bD0bO0bO0be0<",
    "be0be0a30a30``0`Z0[F0U10Sl0SY0SY0SY0SY0S]0Sa0Sd0Sd0Se0Sk0T30T70TT",
    "D0TH0TJ0TQ0TU0T^0U00U30U>0UQ0Ud0V@0VH0VJ0V_0X70ZE0a30b?0b@0bL0bkf",
    "0cI0d00e90l10n?0nW0nc0o10oX10310310J10V10i11C11\\11i12512P12^13@1h",
    "3@13@1380mG0l[0kn0ke0k>0k80jb0jY0jL0jH0ii0i`0iP0iH0iH0iH0i]0i`0i9",
    "i0jG0ja0k00k?0kW0kl0lG0lG0lG09Y09U09S09Q09809209008h08b08_08S08LW",
    "07b06Z05F05004n04m04m04d04d04d04g04g04g04g04g04g04j04j04j04j04k0>",
    "4d04d04d04g04g04g04b04b04b04b04e04e04e04_04\\04\\04\\04c04c04c04^04j",
    "^04^04d04d04d04c04c04a04a04\\04\\04\\04\\04`04a04a04a04a04a04d04d04d5",
    "04d04a04a04a04_04^04K04K04K04O04O04O04R04M04M04Y04\\04`04`04a04_0G",
    "4b04b04b04`04`04`04\\04\\04`04\\04b04e04e04e04g04g04c04c04c04c04c04`",
    "c04c04a04a04a04c04e04e04c04c04g04a04a04d04b04b04d04m04o04o04o04jY",
    "04j04j04m04m04m04m04m04o04e04h04n04l04l04n05005005005005004j04j0B",
    "5005005005005005005105105305304c04c0541=i1=i1=i0<a0<a0<a0>E0?:0?8",
    ":0?:0>:0=d0=b0=V0=F0<n0;c0;c0;_0;_0;e0;e0;I0;@0:U09I08i08Q08I08I7",
    "08I08Z09A09G0;60?<0@B0@B0@E0@G0@G0@Y0@[0@\\0@]0@]0@]0@c0@e0@f0@o0d",
    "A00A20A20A20A80A80A=0A<0A=0A>0AM0AP0AQ0AQ0AT0AU0AU0AQ0@m0@60?90>C",
    "J0>D0>60=o0=W0=W0=W0=S0=S0=B0<F0;Z0;O0;=0;70:m0:e0:e0:e0:e0:^0:^E",
    "0:^0:^0:b0:b0:i0;00;00;00;10;80;;0;;0;;0:k0:S0:O0:N0:N0:L0:L0:S0A",
    ":V0:Z0:a0:e0;80;a0=L0AZ0AZ0AZ0>P0=;0=60=00<o0=00=60=T0@00A?0A?0AZ",
    "?0@k0@k0@k0A50AN0BF0C90CO0CP0CS0CP0CX0CZ0D=0D=0D=0@d0@20@20?k0?kN",
    "0?\\0?Z0?U0?P0?70>]0>]0>]0?F0@F0@F0@F0@F0@F0@D0@?0@?0@40?k0?c0?X0j",
    "?U0?U0?U0?a0@C0AV0DQ0DQ0DQ0DJ0CX0CX0CX0CX0CJ0B\\0B\\0B\\0C40E_0FN0Fk",
    "^0Fg0Fh0G10H50H50H50Fl0FW0FM0F;0Ed0DW0BP0A20A10@e0@e0@d0@`0@`0@\\0",
    "0@\\0@[0@S0@S0@M0@M0@I0@I0@I0@U0Bj0C?0C@0C[0CG0CG0CG0Ag0;10:009T0o",
    "9S09O09S09S09Y09b0:;0;L0?V0AT0B=0BA0BG0B_0Bb0CG0CG0CG0CK0C;0BP0Bm",
    "P0AD0A80@T01c01a01a01`01a01_01_01T01T01R01R01R01U01V01V01U01S01S2",
    "01S01X01X01X01E01E01E01F01F01F01E01E01E01H01H01H01E01E01E01E01E0Q",
    "1@01@01101:01@01A01=01@01A01A01N01N01N01L01L01L01O01W01W01S01P01o",
    "R01R01R01P01P01P01K01E01E01C01B01B01B01B01B01=01=01=01A01J01J01JC",
    "01H01H01H01H01H01H01H01I01I01I01D01B01B01B01D01D01D01C01C01C01G0N",
    "1J01J01J01<01<01<01@01I01Q01I01I01N01N01N01N01N01N01R01R01R01P01n",
    "P01P01T01W01W01U01T01Q01Q01P01Q01Q01Q01Q01Q01T01_01_01b01b01_01dP",
    "01d01c01^01c01c01d02H03?04:04W04o05D05M05O05O05Q05Q05V05W05W05V0J",
    "5L05H05H05L05L05F05?05?05=05;05;05;05105104n04n04l04l04f04f04a04O",
    "a04\\04a04a04\\04Y04Y04Y04X04X04X04X04X04K04K\\",

#else
    "GD0000108000",
    "00P",
    "1gT6R",
    "0nS0na0o00oI0oV0o_10510910<10U1151151150Yn0Yn0Yn0Yl0Yg0Yg0YL0YF0`",
    "Y?0Y=0Y30Y20X\\0X\\0X\\0XX0XU0XM0XB0X50X50X20X20Wm0Wm0Wm0X30X70XA0Xb",
    "R0X`0Xi0Y;0YN0YV0Z00ZA0ZI0ZY0Zb0[50[C0[U0[f0\\60\\A0\\X0\\X0\\X0Z70XnM",
    "0X:0X60Wj0VU0S`0Rk0R30QF0P90O80O00Na0NU0NM0N40Mn0Mg0M]0M[0MZ0MK0f",
    "MF0MB0M<0M20Lo0Lo0Ln0Li0Li0Li0Lg0Lc0La0La0L_0L_0L_0L`0Lg0Lg0Li0Li",
    "k0M<0M=0M=0M=0M=0M=0M=0MO0MV0MV0MV0MS0MS0MS0M\\0Mo0N20N60N>0Na0OcK",
    "0VA0YE0YJ0YN0YN0YU0YX0YX0YX0YG0Y40Xj0Xh0TN0Ri0R@0Qc0Qc0Qd0Qd0R801",
    "RV0Ro0SG0T40TJ0TM0TQ0TW0TX0UB0V10V10V10UQ0U50U40U30To0U30U40UB0US",
    "B0UB0S10QG0Pg0P`0PG0PG0PG0PP0PT0PX0P`0Pj0QK0Q\\0Qh0RD0RS0RS0RS0Ql[",
    "0Pd0Pd0Pd0Q:0Q>0R90Sd0T30T?0TU0U20UR0Vk0Vk0Vk0PU0NU0Mg0Me0Ma0M\\0A",
    "MT0MO0MO0MO0MO0MO0MO0MN0MK0MK0MI0MI0MI0MN0MN0MO0MU0MU0MU0MU0MU0M?",
    "N0MN0MQ0MN0MM0MD0M?0M?0M?0MM0MM0MM0MA0M@0M@0MA0MM0Pm0SN0SW0SW0SWK",
    "0SM0SL0SL0SM0SO0SV0Sf0Sf0Sh0T10T\\0YG0Zc0Zc0Zc0Z70Y40Wj0U70U70U70U",
    "V>0V>0V>0V>0V>0VB0VH0V[0Vn0Vn0W70W;0W<0W<0W?0W@0W@0W@0W@0Vm0VH0Ua",
    "_0U[0U[0U[0VK0VK0VK0Un0U60SF0RJ0RJ0RJ0S70Ug0WQ0Ye0ZH0ZL0ZR0[X0_<[",
    "0a70a;0a>0a@0a>0a>0`O0^h0^h0^h0^V0^V0^V0^V0^V0aG0aj0b50bG0bG0cA0G",
    "cA0cA0a<0a<0`]0`\\0[10Tn0T00SZ0SS0SS0SS0SY0S`0Sb0Sc0Sh0Sh0Sj0T50Tf",
    "50TF0TG0T^0T^0T_0Td0U80UI0UJ0UW0Ui0V40VB0Vk0X;0ZC0`I0ai0bM0bZ0bmM",
    "0c?0c\\0eB0lK0nN0n\\0nf0nl0oD0om10B10E10Y10Y11811X11m12<12Q12b12n1L",
    "361361360nX0l50ko0kf0k]0kT0kD0jS0jR0jA0ik0ii0iO0iM0iM0iM0iU0ie0j?",
    "10jH0jR0jk0kJ0k[0l80lF0lF0lF09^09U09I09I09508n08d08c08^08W08P086X",
    "07]07005E04d04d04]04]04^04^04]04\\04\\04\\04`04a04a04`04Y04Y04Y04Y0B",
    "4S04\\04\\04\\04a04a04a04U04U04U04X04Y04Y04Y04W04W04W04W04V04S04S04\\",
    "Q04S04U04U04U04U04U04U04U04R04U04Z04U04Z04Z04S04S04W04S04S04S04Sn",
    "04S04T04T04T04N04N04N04L04L04L04L04L04L04N04N04N04O04O04O04Y04Y0^",
    "4Y04Y04Y04[04[04[04W04W04W04W04W04W04U04U04W04U04U04R04R04R04V045",
    "Z04Z04X04[04[04X04[04[04Z04Z04[04[04[04[04X04X04X04_04_04_04_04\\4",
    "04\\04\\04Z04Z04\\04\\04\\04_04`04`04_04_04`04a04c04f04c04f04i04l04l0E",
    "4l04_04_04_04m04m04m04i04i04i04i05905C1=n1=k1=k0<X0<X0<X0>L0?60?n",
    "60?60>^0>90=f0=f0=T0=C0<30<00;e0;]0;\\0;V0;L0:k0:B09]08\\08M08E08EA",
    "08E08X08j09M0;F0?20@70@80@C0@I0@P0@R0@T0@U0@[0@_0@_0@`0@f0@f0@o01",
    "@o0A20A20A30A20A70A<0AB0AB0AE0AJ0AL0AO0AZ0AZ0AZ0AR0AL0A10@80?60>S",
    "S0><0>20=Y0=W0=T0=S0=R0=R0=H0<M0;Y0;S0;:0;30;00:h0:e0:_0:_0:_0:_i",
    "0:^0:^0:^0:^0:l0:l0:l0:l0:n0;50;90;90;90:j0:]0:W0:U0:Q0:N0:Q0:Q0@",
    ":U0:W0:X0:g0;>0;n0=n0Aa0Aa0Aa0>O0=B0<i0<i0<i0<k0<n0=\\0@n0AF0AF0AH",
    "F0@m0@l0@l0@l0AV0Bh0C:0C;0CL0CL0CL0CT0Ce0D<0D<0D<0@o0@60@20?g0?_Z",
    "0?W0?W0?V0?P0>l0>W0>W0>W0?G0@90@?0@B0@B0@B0@B0@B0@;0?o0?f0?R0?R0A",
    "?R0?R0?W0?[0@60AZ0E60E60E60DL0C[0C[0C[0C[0C[0BZ0BZ0BZ0Bg0EB0FH0F2",
    "J0F[0F_0F_0Gd0Gd0Gd0G;0F_0FH0El0ET0DV0Al0A30@g0@c0@^0@^0@^0@^0@^0",
    "0@Z0@W0@Q0@Q0@Q0@P0@N0@O0@O0@O0BO0C:0C>0CH0CM0CM0CM0AE0;B0:009_0N",
    "9K09E09E09E09]0:20:U0;;0>f0A80BG0BG0BG0BL0Bg0Bg0Bg0Bi0B60A^01G01Q",
    "G01G01I01N01O01S01_01_01d01_01_01_01Q01M01M01M01R01S01S01R01L01LE",
    "01L01V01V01V01I01I01I01J01K01K01K01K01D01C01C01=01=01;01;01;01=0f",
    "1;01<01:01<01<01<01A01D01I01I01I01K01I01I01H01K01H01J01J01N01S01H",
    "S01S01S01O01C01C01C01C01E01E01@01=01;01;01;01G01G01G01C01@01@01CL",
    "01@01>01M01L01L01Q01Q01L01G01G01G01G01H01H01F01E01E01<01<01<01;0Q",
    "1;01?01K01K01K01K01E01E01E01E01E01E01E01V01V01J01J01J01J01L01N01M",
    "N01P01P01P01P01Y01Y01Y01V01V01S01V01`01`01X01`01f01X01X01O01O01ON",
    "01S01_01a01a01d02002802_04504j05B05C05E05F05K05O05R05R05R05I05I0:",
    "5I05L05L05L05?05905405405705705405105004o04n04n04h04e04`04`04`04m",
    "`04`04U04U04U04Z04Z04Z04Q04O04O04O04R04R04Ri",

#endif

#if 0

#if 1
    "GD0000076801",
    "00P",
    "0T8<h",
    "00000000000004h04c04_04_04U04T04T04T04T04M04X01J01J01J01Z01Z01^05",
    "1Z01W01U01U01S01N01N01N02j03g03g03h03g03b03^03^03]03]03W03W03Q03g",
    "Q03801J01J01J01N03e03e00000000000004I04;03Z01G01:01:01801:01:01<o",
    "01T03e04N04R04U04U0010010010010000010050080080070070080080080010h",
    "0000100000000000100800800800800800800700000000000024C24223d23Q23n",
    "722e22P22<22821`21W21W21X21g21g20V1om1ol1ol1ol1ol20520522H22\\22\\5",
    "22\\24?24b0000000000000000000000000000070070000000070070000000FV0E",
    "FM0FI0FI0FJ0FM0FK0FR0FR0FU0F_0Fa0Ff0G00G00000QD0P[0P?0O`0OI0NM0NU",
    "10Ma0M70L\\0LM0Ko0Ko0KQ0KQ0GI0FW0ET0E80De0De0De0Dh0EC0F=0FX0Fn0GB@",
    "0GK0GO0G[0G[0G[0G[0GX0GW0GW0GW0GX0G^0Gj0Gj0Gj0;90;70;50;50;30:m0k",
    ":m0:m0;30;?0;G0;L0;L0I80IN0IN0IY0I^0I_0I_0Im0In0JE0JE0JE0JE0JM0J8",
    "M0nH0nH0n;0n40mm0mm0n00n30n30n300618O18O18O18O19_19_006006136132m",
    "12o12o12o13213713D13W13f14E14_15D17617G17G17G0F20F00F00F00F>0FO0J",
    "In0In0In0Ig0Ia0I`0IS0IG0IC0IA0I<0I40I40Hi0H`0HT0HS0HO0HG0H@0H<0HS",
    "80H40Gi0Ge0G\\0GY0GM0GJ0FE0En0E\\0E\\0EQ0EP0EQ0EY0EY0Ea0Ea0Ek0Eh0Ehk",
    "0F30G60GO0Gl0Gl0Gb0GW0G_0GW0GI0FP0FP0EF0FP0EL0EL0EL0EL0EL0CF0C@0K",
    "C?0C<0C;0C20C20C10C10Bo0Bm0Bl0Bj0Bj0Bj0Bj0Bb0Bb0Bb0Bb0B_0B_0B_0Bd",
    "_0B\\0B\\0BZ0B\\0B\\0B\\0B\\0B\\0BZ0BZ0BZ0BU0BQ0BM0BB0B?0B?0B?0BN0BT0BU>",
    "0BX0BZ0BX0BX0BZ0B^0B^0B^0B^0B^0B^0B]0B^0B`0Be0B`0[a0[g0[g0[j0[n0N",
    "\\20\\40\\C0\\C0\\L0\\L0\\\\0\\\\0\\b0\\d0\\d0060080081cZ1cJ1cJ1eY1eY1d01d01`S",
    "F1`F1`H1`f1`j1aI1aQ1a_1af1b61bR1bR1bX1bM1aC1aC0060060060061UB1SjU",
    "1Sj1R<1QF1O[1NL1M01L71Jb1IT1HA1G81F51E?1D:1Bc1Am1@a1?h1=i1=11<G1L",
    ";Q1:G19R18j18417I16`16516516515@2P]12]12]12]12=0021000oM0oM0o20o6",
    "20nU0m[0lM0k[0k[0kZ0kB0iG0f20e^0e]0eO0eO0eO0eK0eI0680670660640646",
    "06406406005o06205o05o05o05k05k05f05f05d05d05`05`05]05Z05Y05U05R0T",
    "5S05S05N05I05P05I05I05J05J05J05N05J05L05L05L05L05L05L05L05L05G05j",
    "A05@05@05A05A05A05D05D05F05F05F05H05H05H05H05H05F05F05F05F05F05Eb",
    "05E05E05E05E05H05H05H05=05=05=05>05>05>05?05?05A05A05B05A05B05B0M",
    "5B05>05A05?05>05>05>05805805705705805705705705705305305305305305Y",
    "405605405605705605705705605305205205205905;05;05;05<05=05H05H05Hb",
    "05H05J05L05L05Q05Q05Q05b05b05d05g05o06206I06h08f0:=0:>0:B0:D0:H05",
    ":M0:T0:V0:[0P>0PL0T10T10Yo00000500000000000000000000000000000000K",
    "000000000000000000000000800800600600500600600100100500100800805n8",
    "05H05505504n04n01=01901901901<01C01F02004d04m04m0010010010010060O",
    "0601401401401M02Z02m03303303302j02d02]02\\02L02@02301`01\\01F01F011",
    "F01F01I01P01Q01R01R01R01R01N01E01E01E03]03`03b03b03`03`03^03^03^m",
    "03b5",
#elif 0
    "GD0000076801",
    "00P",
    "0T:Ao",
    "00000000000104g04g04g04g04_04_04Q04Q04Q04M04Q01M01M01M01W01W01Z0M",
    "1Z01W01U01U01I01I01I01I02j03a03d03d03d03]03]03Z03T03T03X03S03R03`",
    "Q03301D01D01D01K03`03`00000000000000004E03[01@01?01?01?01=01=01=J",
    "01M03g04P04X04[04[0000000000010010010050080080070070080080080080<",
    "0000000000000000000800800800800800800700000000000024F23b23Z23A23D",
    "@22`22S22A22521`21P21P21`22C22C20L2001oo1oo1oo1oo20120122J22`22`;",
    "22`24G24j0000000000000000000000000000070070000000070070000000FV0Y",
    "FN0FM0FM0FM0FN0FP0FQ0FR0FX0FX0Fa0Fe0G30G30000QG0Q40PA0O]0O@0N]0MI",
    "h0Mf0Lm0LY0LV0L10L10KT0KT0GS0F^0EW0E<0Dk0Dh0Dh0Dh0EB0F=0FW0Fo0G@d",
    "0GF0GI0GZ0G^0G^0G[0G^0G[0GY0G[0G^0G`0Ga0Ga0G`0;=0;=0;00:n0;00;00_",
    ";00;00;:0;;0;D0;Q0;Q0I?0IJ0IJ0IZ0IZ0I\\0I]0J00J20J?0JD0JQ0JT0JT00K",
    "00nV0nN0n=0n70ml0ml0ml0mm0n30n300618X18T18T18T19]19]000000137134f",
    "13112o12o13113=13E13[13f14C14X15E17717I17I17I0F10Ej0Ej0Ej0FD0FM0D",
    "Ij0Ij0Ij0Ij0I_0I[0IP0IM0IF0I@0I?0I40Hm0Hf0H]0HU0HS0HO0HL0HF0H@0HM",
    "40Gl0Gb0Gb0Gb0GT0GN0GD0FE0En0EV0EV0EQ0EN0EQ0EY0Eb0Eh0Eh0Ei0Ek0Ek]",
    "0F40G90GK0Gj0Gj0Gd0GW0Ga0GW0GN0FP0FP0EJ0FP0EI0EI0EI0EI0EI0CF0C?0D",
    "C>0C:0C60C60C30Bm0Bj0Bi0Bg0Bg0Bg0Bf0Be0Bd0Bd0Bd0Bd0B`0Bc0B`0B`0B;",
    "`0B^0B\\0B[0B\\0B\\0B[0B[0BW0BR0BR0BT0BT0BR0BO0B@0B@0B@0BE0BS0BV0BV5",
    "0BV0BV0BV0BW0BW0BZ0B\\0B_0B_0B`0B`0B`0B`0B`0Bb0Bb0[f0[f0[f0[f0[n0E",
    "\\30\\:0\\<0\\A0\\K0\\L0\\S0\\W0\\W0\\_0\\`0\\_0080081cg1cL1cL1eN1eN1dA1dA1``",
    "J1`J1`V1`o1a01a31aQ1ac1ao1bH1bL1bO1bS1bO1a?1a?0000000000001UA1So\\",
    "1So1RM1Q41OZ1NW1M:1Kk1J`1I_1HJ1G<1FB1E<1Cn1Bo1B41@]1?^1=n1<k1<?1f",
    ";I1:H19R18j18517A16f16016016015F2Q612g12g12g12>0020on0oK0oK0nn0nP",
    "n0nQ1[I0lS0kU0kU0kU0kC0iA0em0e_0e]0eT0eR0eL0eK0eK06;0650630630606",
    "06006006006406406406306005m05i05f05f05d05a05^05^05[05Y05X05W05R0X",
    "5P05P05P05P05J05J05J05H05H05G05N05G05N05H05H05G05H05F05F05F05F05=",
    "F05F05E05D05B05B05B05A05B05E05G05H05J05J05J05N05N05J05J05J05J05JW",
    "05H05H05G05A05A05=05=05=05=05=05>05>05>05=05=05=05?05?05?05?05?0_",
    "5>05=05=05=05=05905905405405405505505505505505505505505705605705:",
    "605605605705705905905805705705605505405405405805=05=05>05B05E05Ed",
    "05E05H05H05Q05Q05S05S05W05[05d05g05m06406F06`08h0:20:=0:@0:D0:J0N",
    ":K0:L0:O0:T0PA0PL0So0T00Yn000005000000000000000000000000000000001",
    "000000000000000000000000800800600600500600600100100500100800805o9",
    "05D05405405205401B01=01:01=01>01>01I02804a04a00100100100100104P0:",
    "4P01001001001D02a02j03103103102e02a02[02[02L02A02301c01\\01D01D01m",
    "D01F01G01M01T01T01T01T01N01I01D01D01D03\\03`03`03`03`03^03]03\\03]X",
    "03^1",
#else
    "GD0000076801",
    "00P",
    "0TKml",
    "00000100100104e04_04Y04X04W04V04F04F04F04F04G01F01F01F01W01W01W0M",
    "1W01W01W01S01G01F01F01G02S03X03Y03Y03Y03W03V03V03V03N03L03L03E03]",
    "@03001E01E01E01F03Y03Y00100100100004P04I03P01C01=01:017016016016e",
    "01B03M04@04K04U04]04]00100105]05]05]05]0080080070070080080080000a",
    "0000000100100000000800800800800800800700000000000024B23b23Y23R23R",
    ";22]22V22=21g21R21L21L21Q22<22<20K1on1ok1ok1ok1oh1oh1oh22@22W22Wf",
    "22W2480000000000000000000000000000000070080070080070070000000FO0J",
    "FL0FJ0FJ0FL0FN0FN0FP0FQ0FS0FW0F]0Fc0Fd0Fd0000QA0Pa0PI0Oe0O;0NQ0M5",
    "o0M]0M20L\\0LL0LL0LL0L:0060GV0FV0EN0E70Dh0Df0Df0Df0E@0Ej0FP0Fi0GBH",
    "0GC0GH0GZ0GZ0GZ0GZ0GZ0GZ0GZ0GZ0G]0G]0G`0G`0G\\0;90;90;90;90;00;00P",
    ";00;00;50;60;K0;O0;O0I@0IL0IL0IR0Ia0Ic0Ie0Il0In0J40J80JB0JM0JX0J>",
    "X0nF0nF0n;0n20n10n00n00n00n00n000618N18H18H18H19Q19Q00000013412ml",
    "12m12l12l13113<13@13O13]14<14O15C16m17E17E17E0Eo0Em0Em0Eo0F?0FS0>",
    "Ij0Ij0Ij0Id0Ic0IV0IU0IP0ID0I:0I60I40Hn0H]0HY0HY0HY0HH0HC0H>0H>0Ha",
    "50Gl0Gj0G_0G^0GS0GO0GH0FA0F00ES0ES0EQ0EQ0EQ0EU0Ea0Eb0Eb0Ee0Ee0Ee>",
    "0F20G30GJ0Gc0Gc0Gc0GN0GY0GN0GK0FF0FF0EL0FF0EI0EI0EI0EI0EI0CD0CB04",
    "C<0C60C60C10Bn0Bm0Bn0Bm0Bk0Bk0Bj0Bi0Bf0Bh0Bf0Be0Bc0Bc0Ba0BZ0BY0B;",
    "Y0BY0BY0BY0BZ0BW0BZ0BX0BX0BU0BX0BU0BT0BT0BK0B;0B;0B;0BH0BM0BO0BYL",
    "0BY0BY0BY0BY0BY0BY0B\\0B_0B\\0B\\0B`0B`0B^0Bb0Bb0Bb0[[0[`0[`0[a0[j0j",
    "\\40\\:0\\<0\\B0\\E0\\K0\\X0\\Y0\\Y0\\[0\\[0\\[0080081cU1cE1cE1eR1eR1dE1dE1`G",
    "71`71`U1`b1`h1a91aH1a_1ah1b61b?1bB1b[1b[1a:1a:0000000000001UK1Sdn",
    "1Sd1RN1Q;1OV1NJ1Lf1Kh1J\\1IU1H:1G11Ej1E11Cm1Bl1B01@]1?X1=a1=41<81U",
    ";@1:F19K18n17o17?16X15m15m15m15?2Pb12\\12\\12\\12:0000oh0oD0oD0nh0nB",
    "h0nK0m_0lS0kV0kV0kU0kD0iE0ef0ea0eZ0eV0eQ0eH0eH0eF06A065064064064Q",
    "06406406206206006005k05i05g05i05i05f05a05d05^05\\05\\05Q05Q05Q05T00",
    "5T05T05P05P05L05K05K05J05K05O05O05G05O05N05E05D05D05D05D05B05B05H",
    "B05@05@05@05<05?05?05?05D05D05D05E05E05E05H05H05H05H05F05E05E05FE",
    "05F05E05@05@05<05<05<05@05@05@05=05=05=05@05@05@05@05B05<05<05@0Y",
    "5@05<05<05>05;05;05;05905905605605805505505805805405605805705605T",
    "605705705605905905;05705705405405405505605705905<05<05=05B05E05Ij",
    "05I05I05K05K05N05X05Z05Z05Z05]05d05n06706H06X08k0:;0:?0:?0:?0:B0O",
    ":Q0:Q0:R0:V0PA0PB0Sn0T500600600700600000000000000000000000000000h",
    "000000000000000000000000800800800800800600606W06W06W06W00800805^[",
    "05905805504o03X01:01601601601601901J02804_04_04`04`04`04_04`04V0l",
    "4V00j00j00j01G02X02`02k02k02k02d02b02V02Q02602201i01g01Z01?01=010",
    "=01?01L01P01P01S01S01S01S01S01901901903R03d03d03d03`03V03V03V03\\e",
    "03\\o",
#endif
#endif
  };

  // GD 情報を登録
  ScipHandler::RawRecvData raw_data;
  raw_data.type = ScipHandler::GD;

  size_t lines_n = sizeof(gd_lines) / sizeof(gd_lines[0]);
  for (size_t i = 0; i < lines_n; ++i) {
    raw_data.lines.push_back(gd_lines[i]);
  }

  ScipHandler scip;
  std::vector<long> points_length;
  scip.convertToLength(points_length, raw_data);

  // PP 情報を登録
  // PP
  // 00P
  // MODL:UHG-08LX(Hokuyo Automatic Co., Ltd.);h
  // DMIN:30;5
  // DMAX:11000;F
  // ARES:1024;
  // AMIN:0;?
  // AMAX:768;6
  // AFRT:384;6
  // SCAN:900;h
  SensorParameter parameter;
#if 0
  parameter.area_total = 1024;
  parameter.area_front = 384;
  parameter.area_min = 30;
#else
  parameter.area_total = 1440;
  parameter.area_front = 540;
  parameter.area_min = 23;
#endif

  UrgCtrl urg;
  urg.setSensorParameter(&parameter);

  size_t points_n = points_length.size();
  for (size_t i = 0; i < points_n; ++i) {
    long length = points_length[i];
    if (length < parameter.area_min) {
      continue;
    }

    double radian = urg.index2rad(i);
    long x = static_cast<long>(length * cos(radian));
    long y = static_cast<long>(length * sin(radian));

    printf("%ld\t%ld\n", x, y);
  }

  return 0;
}
