#ifndef URG_CTRL_H
#define URG_CTRL_H

/*!
  \file
  \brief URG 制御ライブラリ

  \author Satofumi KAMIMURA

  $Id: UrgCtrl.h 196 2008-09-02 04:04:09Z satofumi $

  \todo IsUsbCom を継承した単一のクラスを作るように変更する
*/

#include "RangeSensor.h"
#include "CaptureMode.h"


namespace qrk
{
  class SensorParameter;
  class ScipHandler;

  /*!
    \brief SCIP2.0 規格による URG シリーズセンサ制御
  */
  class UrgCtrl : public RangeSensor
  {
    UrgCtrl(const UrgCtrl& rhs);
    UrgCtrl& operator = (const UrgCtrl& rhs);

    struct pImpl;
    std::auto_ptr<pImpl> pimpl;

  public:
    enum {
      DefaultBaudrate = 115200, //!< 標準の接続後ボーレート

      VEND = 0,                 //!< ベンダ情報
      PROD,                     //!< 製品情報
      FIRM,                     //!< ファームウェアバージョン
      PROT,                     //!< プロトコルバージョン
      SERI,                     //!< センサ固有シリアル番号

      UpsideDown = true,        //!< センサが天地逆に配置
      Normal = false,           //!< センサが正常に配置

      On = true,                //!< オン
      Off = false,              //!< オフ

      Infinity = 0,             //!< 無限回
    };


    /*!
      \brief capture() のブロッキング設定
    */
    typedef enum {
      Blocking,              //!< １スキャンデータ取得までブロッキング
      NonBlocking,   //!< 新規データがない場合に capture() は 0 を返す
    } BlockingMode;

    UrgCtrl(void);
    ~UrgCtrl(void);

    const char* what(void);

    bool connect(const char* device, long baudrate = DefaultBaudrate);


    /*!
      \overload

      \brief 接続

      引数を解析しての接続処理。以下のオプションを解析可能

      - --urg_port=[port] ... 接続ポートを指定
      - --urg_baudrate=[port] ... 接続ボーレートを指定
      - --urg_handstand ... URG の取得データの配置を開始と終了で反転させる

      \retval true 接続成功
      \retval false 接続に失敗
    */
    bool connect(int argc, char *argv[]);


    /*!
      \overload
    */
    bool connect(Connection* con);

    Connection* connection(void);

    ScipHandler* scipHandler(void);

    void disconnect(void);

    bool isConnected(void);


    /*!
      \brief 取得モードの設定

      \param[in] mode 取得モード設定 (AutoCapture / ManualCapture)

      <strong>Example</strong>
      \code
UrgCtrl urg;
urg.setCaptureMode(AutoCapture); \endcode

      取得モードの詳細は \ref compare_auto_manual_page を参照のこと。
    */
    void setCaptureMode(CaptureMode mode);


    /*!
      \brief "取得モード" の取得

      \return 取得モード
    */
    CaptureMode captureMode(void);


    bool setHighSensitiveMode(bool on = true);


    /*!
      \brief レーザの出力制御

      \param[in] output On / Off を指定する

      \todo 使用例を配置する
    */
    void setLaserOutput(bool output = On);


    /*!
      \brief capture() のブロッキング処理の変更

      \param[in] mode ブロッキングモード
    */
    void setBlockingMode(BlockingMode mode);


    /*!
      \brief capture() のブロッキング設定の取得
    */
    BlockingMode blockingMode(void);

    int capture(long data[], size_t data_max);

    int intensity(long data[], size_t data_max);


    /*!
      \brief データ取得の開始
    */
    bool startCapture(int times = Infinity);


    /*!
      \brief データ取得の開始
    */
    void stopCapture(void);

    bool setTimestamp(int ticks = 0);

    int timestamp(void);


    // 角度変換
    double index2rad(const int index) const;
    int rad2index(const double radian) const;


    /*!
      \brief 取得バッファ数の設定

      内部に格納できる周期データ数

      \param[in] size バッファ数

      \todo 概念図を用いて説明する
    */
    void setMaxBufferDepth(size_t size);
    // size_t maxBufferDepth(void);


    /*!
      \brief いくつのデータをグループ化するかの設定

      \param[in] group データをグループ化する数

      \todo ちゃんと説明する

      \attention 設定を変更すると、データ取得は中断される
    */
    void setSkipLines(size_t lines);


    /*!
      \brief 取得フレームの間引き回数設定

      \todo 説明する

      \attention 設定を変更すると、データ取得は中断される
    */
    void setFrameSkips(void);


    /*!
      \brief 取得幅の設定

      \todo 説明する

      \attention 設定を変更すると、データ取得は中断される
    */
    void setCaptureRange(int first, int last);

    long minDistance(void) const;

    long maxDistance(void) const;

    int maxBufferSize(void) const;
    int maxScanLines(void) const;


    /*!
      \brief 測距の周期時間を取得

      \return 測距の周期時間
    */
    size_t scanMsec(void);


    bool versionLines(std::vector<std::string>& lines);


    /*!
      \brief センサパラメータの取得

      \todo 説明する
    */
    void sensorParameter(SensorParameter* parameters);


    /*!
      \brief センサパラメータの設定

      \todo 説明する

      \attention 設定を変更すると、データ取得は中断される
    */
    void setSensorParameter(const SensorParameter* parameters);


    /*!
      \brief 取得データを反転させる

      \param[in] on 反転させるとき true

      \todo 説明がわかりにくい。図があった方がよい

      \attention 設定を変更すると、データ取得は中断される
    */
    void setUpsideDown(bool state = UpsideDown);
  };
};

#endif /* !URG_CTRL_H */
