/*!
  \file
  \brief URG 制御ライブラリ

  \author Satofumi KAMIMURA

  $Id: UrgCtrl.cpp 212 2008-09-04 03:48:22Z satofumi $

  \todo 取得データを、固定サイズのデータ配列に展開してから格納するべき
  \todo now_capture_ のときは、いったん停止してから、コマンドを発行させる
  \todo 排他制御を適切に行う
  \todo エラーコードに適切な名前を付ける
*/

#include "UrgCtrl.h"
#include "SensorParameter.h"
#include "ScipHandler.h"
#include "ConnectionUtils.h"
#include "SerialCtrl.h"
#include "Thread.h"
#include "Lock.h"
#include "LockGuard.h"
#include "Semaphore.h"
#include "delay.h"
#include <cmath>

// !!! 送受信データの記録用
//#include "ConnectionRecorder.h"

using namespace qrk;


struct UrgCtrl::pImpl
{
  enum {
    ScipTimeout = 100,   //!< 前回のデータ取得読み出し用のタイムアウト
  };
  std::string error_message_;
  Thread capture_thread_;
  Connection* con_;
  ScipHandler* scip_;
  SensorParameter sensor_parameter_;
  CaptureMode capture_mode_;
  BlockingMode blocking_mode_;
  bool upside_down_;
  int capture_times_;
  ScipHandler::RawRecvData raw_data_;
  size_t last_timestamp_;

  Lock urg_mutex_;
  bool now_capture_;
  Semaphore captures_sem_;

  std::vector<long> intensity_data_;

  size_t skip_lines_;


  pImpl(void)
    : error_message_("no error."),
      capture_thread_(capture_function, this),
      con_(NULL), scip_(NULL),
      capture_mode_(ManualCapture), blocking_mode_(Blocking),
      upside_down_(Normal), capture_times_(0), last_timestamp_(0),
      now_capture_(false), captures_sem_(0),
      skip_lines_(1)
  {
  }


  ~pImpl(void)
  {
    delete con_;
    delete scip_;
  }


  void initializeSerial(void)
  {
    if (! con_) {
      //con_ = new ConnectionRecorder(new SerialCtrl);
      con_ = new SerialCtrl;
    }
  }


  bool connectToUrg(long baudrate)
  {
    // scip_ の初期化
    if (! scip_) {
      scip_ = new ScipHandler;
    }
    scip_->setConnection(con_);

    // ボーレート合わせ
    if (! adjustBaudrate(baudrate)) {
      return false;
    }

    // PP パラメータの取得
    if (! scip_->parameters(&sensor_parameter_)) {
      error_message_ = scip_->what();
      return false;
    }

    return true;
  }


  bool adjustBaudrate(long baudrate)
  {
    long try_baudrates[] = { 115200, 19200, 38400, };
    size_t try_size = sizeof(try_baudrates) / sizeof(try_baudrates[0]);

    // 接続したいボーレートを配列の先頭と入れ換える
    for (size_t i = 1; i < try_size; ++i) {
      if (baudrate == try_baudrates[i]) {
        std::swap(try_baudrates[0], try_baudrates[i]);
        break;
      }
    }

    // 指定のボーレートで接続し、応答が返されるかどうか試す
    for (size_t i = 0; i < try_size; ++i) {

      // ホスト側のボーレートを変更
      if (! con_->setBaudrate(try_baudrates[i])) {
        error_message_ = con_->what();
        return false;
      }

      // QT の発行
      int reply = scip_->setLaserOutput(ScipHandler::Off, ScipHandler::Force);

      if ((reply < 0) && (reply != -14)) {
        // MD/MS コマンドを受け取ると、ここの分岐に入る
        // 受信内容を全て読み飛ばしてから、次の処理を行う
        // (reply == -14) のときは、SCIP1.1 応答で 'E' の場合
        skip(con_, ScipTimeout);
        reply = 0x00;
      }

      // 応答が返されれば、既に SCIP2.0 モードであり "SCIP2.0" の発行は不要
      if (! (reply == 0x00)) {

        int scip_reply = scip_->sendScip20();
        if (scip_reply == -0x0E) {
          // SCIP1.1 の場合は、エラーメッセージを出力して終了させる
          error_message_ =
            "SCIP1.1 is not supported. Please update URG-04LX firmware.";
          return false;
        }
        if (scip_reply < 0) {
          // 応答がなければ、違うボーレートに接続したものとみなす
          continue;
        }
      }
      if (baudrate == try_baudrates[i]) {
        return true;
      }

      // URG 側を指定されたボーレートに変更する
      if (! scip_->changeBaudrate(baudrate)) {
        error_message_ = scip_->what();
        return false;
      } else {
        return true;
      }
    }
    error_message_ = "fail baudrate detection.";
    return false;
  }


  void disconnect(void)
  {
    stopCapture();
    if (con_) {
      con_->disconnect();
    }
  }


  int capture(long data[], size_t data_max)
  {
    if (capture_mode_ == ManualCapture) {
      return captureByManual(data, data_max);

    } else {
      return captureByAuto(data, data_max);
    }
  }


  int intensity(long data[], size_t data_max)
  {
    // !!! 適切に排他処理を行うこと

    // 格納済みの強度データをコピーして返す
    size_t n = intensity_data_.size();
    if (n > data_max) {
      n = data_max;
    }

    for (size_t i = 0; i < n; ++i) {
      // !!! copy 系のアルゴリズムを使うべき
      data[i] = intensity_data_[i];
    }
    return n;
  }


  std::string createCaptureString(CaptureMode mode)
  {
    // !!! 設定されている場合は、MS/GS が使えるようにする

    // !!! ここでの範囲指定は、ユーザ指定のものに置き換えられるようにする
    // !!! そのためには、index2deg() 用に、取得データ範囲を保持する必要がある
    // !!! 取得データ範囲を保持する変更は、未実装

    // !!! 可能ならば、取得開始は 0 からでなく、area_min からにすること
    char buffer[] = "MDbbbbeeeessftt\r";
    if (mode == ManualCapture) {
      // GD/GS
      snprintf(buffer, 14, "G%c%04d%04d%02d\r",
               'D',
               0, //sensor_parameter_.area_min,
               sensor_parameter_.area_max,
               skip_lines_);

    } else {
      // MD/MS/ME
      char command_ch = (mode == IntensityCapture) ? 'E' : 'D';
      snprintf(buffer, 17, "M%c%04d%04d%02d%d%02d\r",
               command_ch,
               0, //sensor_parameter_.area_min,
               sensor_parameter_.area_max,
               skip_lines_,
               0,
               capture_times_);
    }
    return buffer;
  }


  // データ送信コマンドの送信
  bool sendCaptureCommand(CaptureMode mode)
  {
    std::string send_string = createCaptureString(mode);
    size_t send_size = send_string.size();
    int n = con_->send(send_string.c_str(), send_size);
    if (n != static_cast<int>(send_size)) {
      // !!! エラーメッセージの更新
      // !!!
      return false;
    } else {
      return true;
    }
  }

  // MD/MS による距離データの取得
  static int capture_function(void* args) {
    pImpl* obj = static_cast<pImpl*>(args);

    while (1) {
      ScipHandler::RawRecvData raw_data;
      if (! obj->scip_->recvCaptureData(raw_data)) {
        // 距離データが受信できなければ、中断する
        obj->error_message_ = obj->scip_->what();
        if (obj->capture_mode_ == IntensityCapture) {
          // ME でデータ取得に失敗したら、未対応とみなして MD データ取得にする
          obj->capture_mode_ = AutoCapture;
        }
        break;
      }

      if (raw_data.type == ScipHandler::Mx_Reply) {
        // 受信したのが、MD/MS の応答パケットならば、再度データを待つ
        continue;
      }

      if (! ((raw_data.type == ScipHandler::MD) ||
             (raw_data.type == ScipHandler::MS) ||
             (raw_data.type == ScipHandler::ME))) {
        // MD/MS/ME 以外の距離データの場合は、中断する
        break;
      }

      LockGuard guard(obj->urg_mutex_);

      // 取得したデータを保管しておき、セマフォ(データ数)を加算する
      std::swap(obj->raw_data_, raw_data);
      if (obj->captures_sem_.value() == 0) {
        // !!! データ数は１以上にならないようにする
        // !!! 複数データを保持する機能は未実装
        obj->captures_sem_.post();
      }

      delay(1);
    }

    LockGuard guard(obj->urg_mutex_);
    obj->now_capture_ = false;

    // !!! 強制的に消灯する関数を pImpl 内で定義すべき
    obj->scip_->setLaserOutput(ScipHandler::Off);

    return 0;
  }


  // ManualCapture のとき
  int captureByManual(long* data, size_t data_max)
  {
    // レーザの点灯指示
    scip_->setLaserOutput(ScipHandler::On);

    // データ取得コマンドの送信
    if (! sendCaptureCommand(ManualCapture)) {
      return -4;
    }

    // データ取得が完了してから、結果を返す
    if (! scip_->recvCaptureData(raw_data_)) {
      // !!! エラーメッセージの更新
      // !!!
      // !!! scip_->what() でメッセージを返せるようにする
      return -3;
    }

    // 取得データをユーザに返す
    return recvUrgData(data, data_max);
  }

  // AutoCapture のとき
  int captureByAuto(long* data, size_t data_max)
  {
    LockGuard guard(urg_mutex_);

    // データ取得中でなければ、データ要求を行う
    if (! now_capture_) {
      if (! sendCaptureCommand(capture_mode_)) {
        // !!! エラーの場合
        return -4;
      }
      now_capture_ = true;

      // データ取得待機スレッドの起動
      capture_thread_.run(1);

      // データは次の capture() で取得させる
      return 0;

    } else {

      if (! captures_sem_.tryWait()) {
        // !!! エラーメッセージの更新
        return 0;
      }

      // 取得データをユーザに返す
      return recvUrgData(data, data_max);
    }
  }


  int recvUrgData(long* data, size_t data_max)
  {
    // !!! SCIP のバージョンを、adjustBaudrate() のときに判別して、
    // !!! ここで使うようにすべき
    std::vector<long> length_data;
    if (! scip_->convertToLength(length_data, raw_data_,
                                 ScipHandler::SCIP_20)) {
      // !!! エラーメッセージの更新
      // !!!
      // !!! error_message_ = scip_->what() でよいかを検討する
      return -2;
    }

    // !!! data_n < 0 のときに、scip_->what() で更新する

    // !!! 直前に投げたコマンドでなければ、エラー扱いにする
    // !!! または、データがくるのを待つべき
    // !!! 問い合わせに対して、0-9 の ID をはさむ機能も、可能ならば実装する

    size_t data_n = length_data.size();
    if (data_n > data_max) {
      data_n = data_max;
    }

    // 強度データを取得するモードのとき、取得データを距離と強度に分ける
    if (capture_mode_ == IntensityCapture) {

      std::vector<long> actual_length_data;
      intensity_data_.clear();
      if (! splitMeData(actual_length_data, intensity_data_, length_data,
                        raw_data_.skip_lines)) {
        // !!! エラーメッセージを適切に更新すべき
        fprintf(stderr, "splitMeData fail\n.");
        return -1;
      }
      std::swap(actual_length_data, length_data);
    }

    for (size_t i = 0; i < data_n; ++i) {
      // !!! copy 系のアルゴリズムを使うべき
      data[i] = length_data[i];
    }

    // !!! コピーが無駄なことを考えると、
    // !!! convertToLength() は、データ格納先配列を渡せる設計に変更すべき

    last_timestamp_ = raw_data_.timestamp;

    return static_cast<int>(data_n);
  }


  bool splitMeData(std::vector<long>& length_data,
                   std::vector<long>& intensity_data,
                   const std::vector<long>& dual_data,
                   const int skip_lines)
  {
    // データが空か、奇数個のデータならば処理しない。データは偶数個のはず
    if (dual_data.empty() || (dual_data.size() % 2)) {
      return false;
    }

    // データを length と intensity とに分割する
    for (std::vector<long>::const_iterator it = dual_data.begin();
         it != dual_data.end();) {

      for (int i = 0; i < skip_lines; ++i) {
        length_data.push_back(*it);
        ++it;
      }

      for (int i = 0; i < skip_lines; ++i) {
        intensity_data.push_back(*it);
        ++it;
      }
    }
    return true;
  }


  // HS モードの設定
  bool setHighSensitive(bool on)
  {
    return scip_->setHighSensitive(on);
  }


  // データ取得の中断
  void stopCapture(void)
  {
    if (now_capture_) {
      // データ取得の停止処理
      scip_->sendQT();
      now_capture_ = false;
    }

    // !!! 自動的に、スレッドが停止するはずでも、タイムアウトするようにすべき

    // スレッドの停止
    if (capture_thread_.isRunning()) {
      capture_thread_.wait();
    }
  }
};


UrgCtrl::UrgCtrl(void) : pimpl(new pImpl)
{
}


UrgCtrl::~UrgCtrl(void)
{
  stopCapture();
}


const char* UrgCtrl::what(void)
{
  return pimpl->error_message_.c_str();
}


bool UrgCtrl::connect(const char* device, long baudrate)
{
  // 接続済みだったら、切断。資源がなければ、シリアルを生成
  if (pimpl->con_) {
    pimpl->con_->disconnect();
  } else {
    pimpl->initializeSerial();
  }
  bool ret = pimpl->con_->connect(device, baudrate);
  if (! ret) {
    pimpl->error_message_ = pimpl->con_->what();
    return ret;
  }

  // URG デバイスへの接続
  return pimpl->connectToUrg(baudrate);
}


//! \attention 未実装
bool UrgCtrl::connect(int argc, char *argv[])
{
  static_cast<void>(argc);
  static_cast<void>(argv);

  // !!!
  return false;
}


bool UrgCtrl::connect(Connection* con)
{
  delete pimpl->con_;
  pimpl->con_ = con;

  // URG デバイスへの接続
  return pimpl->connectToUrg(DefaultBaudrate);
}


Connection* UrgCtrl::connection(void)
{
  return pimpl->con_;
}


ScipHandler* UrgCtrl::scipHandler(void)
{
  return pimpl->scip_;
}


void UrgCtrl::disconnect(void)
{
  pimpl->disconnect();
}


bool UrgCtrl::isConnected(void)
{
  if (! pimpl->con_) {
    return false;
  }
  return pimpl->con_->isConnected();
}


void UrgCtrl::setCaptureMode(CaptureMode mode)
{
  stopCapture();
  pimpl->capture_mode_ = mode;
}


CaptureMode UrgCtrl::captureMode(void)
{
  return pimpl->capture_mode_;
}


bool UrgCtrl::setHighSensitiveMode(bool on)
{
  // !!! 未接続のときに、戻る

  stopCapture();
  return pimpl->setHighSensitive(on);
}


void UrgCtrl::setLaserOutput(bool output)
{
  // !!! 未接続のときに、戻る

  if (output) {
    pimpl->scip_->setLaserOutput(ScipHandler::On);
  } else {
    pimpl->scip_->setLaserOutput(ScipHandler::Off);
  }
}


void UrgCtrl::setBlockingMode(BlockingMode mode)
{
  if (pimpl->blocking_mode_ == NonBlocking) {
    // スレッドの停止を指示し、実際に停止するのを待つ
    stopCapture();
  }

  pimpl->blocking_mode_ = mode;
}


UrgCtrl::BlockingMode UrgCtrl::blockingMode(void)
{
  return pimpl->blocking_mode_;
}


int UrgCtrl::capture(long data[], size_t data_max)
{
  // !!! 接続されていなければ、戻る
  // !!! 他の関数にも、この処理を入れること

  return pimpl->capture(data, data_max);
}


int UrgCtrl::intensity(long data[], size_t data_max)
{
  return pimpl->intensity(data, data_max);
}


//! \attention 未実装
bool UrgCtrl::startCapture(int times)
{
  static_cast<void>(times);

  // ManualCapture のときは、スレッドで指定回数 GD を呼び出す
  // !!!

  // AutoCapture のときは、MD コマンドを回数指定で呼び出す
  // !!!

  return false;
}


void UrgCtrl::stopCapture(void)
{
  // !!! 接続確認

  pimpl->stopCapture();
}


//! \attention 未実装
bool UrgCtrl::setTimestamp(int ticks)
{
  static_cast<void>(ticks);

  // タイムスタンプ合わせ
  // !!!

  return 0;
}


int UrgCtrl::timestamp(void)
{
  return pimpl->last_timestamp_;
}


double UrgCtrl::index2rad(const int index) const
{
  return (index - pimpl->sensor_parameter_.area_front)
    * 2.0*M_PI / pimpl->sensor_parameter_.area_total;
}


int UrgCtrl::rad2index(const double radian) const
{
  int index = static_cast<int>((radian * pimpl->sensor_parameter_.area_total)
                               / (2.0*M_PI))
    + pimpl->sensor_parameter_.area_front;

  if (index < 0) {
    index = 0;
  } else if (index > pimpl->sensor_parameter_.area_max) {
    index = pimpl->sensor_parameter_.area_max;
  }
  return index;
}


//! \attention 未実装
void UrgCtrl::setMaxBufferDepth(size_t size)
{
  static_cast<void>(size);

  // バッファ拡張
  // !!!
}


//! \attention 未実装
void UrgCtrl::setSkipLines(size_t lines)
{
  pimpl->skip_lines_ = lines;
  stopCapture();
}


//! \attention 未実装
void UrgCtrl::setFrameSkips(void)
{
  // !!!

  stopCapture();
}


//! \attention 未実装
void UrgCtrl::setCaptureRange(int first, int last)
{
  static_cast<void>(first);
  static_cast<void>(last);

  // !!!

  stopCapture();
}


long UrgCtrl::minDistance(void) const
{
  return pimpl->sensor_parameter_.distance_min;
}


long UrgCtrl::maxDistance(void) const
{
  return pimpl->sensor_parameter_.distance_max;
}


int UrgCtrl::maxScanLines(void) const
{
  return pimpl->sensor_parameter_.area_max + 1;
}


int UrgCtrl::maxBufferSize(void) const
{
  return maxScanLines();
}


size_t UrgCtrl::scanMsec(void)
{
  int scan_rpm = pimpl->sensor_parameter_.scan_rpm;
  return (scan_rpm <= 0) ? 1 : (1000 * 60 / scan_rpm);
}


bool UrgCtrl::versionLines(std::vector<std::string>& lines)
{
  return pimpl->scip_->versionLines(lines);
}


void UrgCtrl::sensorParameter(SensorParameter* parameters)
{
  *parameters = pimpl->sensor_parameter_;
}


void UrgCtrl::setSensorParameter(const SensorParameter* parameters)
{
  pimpl->sensor_parameter_ = *parameters;
}


void UrgCtrl::setUpsideDown(bool state)
{
  pimpl->upside_down_ = state;
}
