#ifndef SCIP_HANDLER_H
#define SCIP_HANDLER_H

/*!
  \file
  \brief SCIP プロトコル処理

  \author Satofumi KAMIMURA

  $Id: ScipHandler.h 195 2008-09-01 22:30:06Z satofumi $
*/

#include "CaptureMode.h"
#include <vector>
#include <memory>
#include <string>


namespace qrk
{
  class Connection;
  class SensorParameter;

  /*!
    \brief SCIP プロトコル処理クラス
  */
  class ScipHandler
  {
    ScipHandler(const ScipHandler& rhs);
    ScipHandler& operator = (const ScipHandler& rhs);

    struct pImpl;
    std::auto_ptr<pImpl> pimpl;

  public:
    typedef enum {
      On,                       //!< 発光中
      Off,                      //!< 消灯中
      StateUnknown,             //!< 不明
    } LaserState;


    typedef enum {
      SCIP_20,
      SCIP_11,
    } ScipVersion;


    /*!
      \brief 受信データのタイプ
    */
    typedef enum {
      TypeUnknown,              //!< 不明
      QT,
      GD,
      GS,
      MD,
      MS,
      ME,
      Mx_Reply,
      Classic_G,
    } CaptureType;

    static const bool Force = true; //!< setLaserOutput 用


    /*!
      \brief 受信データの情報
    */
    typedef struct {
      CaptureType type;         //!< 受信データの種類
      int error_code;           //!< エラーコード
      long timestamp;           //!< タイムスタンプ [msec]
      int capture_first;        //!< 取得開始インデックス
      int capture_last;         //!< 取得終了インデックス
      int skip_lines;           //!< 間引きライン数
      int skip_frames;          //!< データ取得間隔
      int left_times;           //!< データ取得の残り回数
      std::vector<std::string> lines; //!< 取得データ
    } RawRecvData;


    ScipHandler(void);
    ~ScipHandler(void);
    const char* what(void);
    void setConnection(Connection* con);
    Connection* connection(void);

    int sendScip20(void);


    /*!
      \brief URG と接続デバイスのボーレートの変更を行う

      \param[in] baudrateボーレート

      \retval true 正常に変更
      \retval false 変更に失敗

      \attention 通信できるボーレートになっていなければならない
    */
    bool changeBaudrate(long baudrate);


    bool versionLines(std::vector<std::string>& lines);
    bool parameters(SensorParameter* parameter);
    int setLaserOutput(LaserState output, bool force = false);

    bool recvCaptureData(RawRecvData& data);


    // 変換した距離データを追加して返す
    bool convertToLength(std::vector<long>& length, const RawRecvData& data,
                         ScipVersion scip_version = SCIP_20);

    bool sendQT(void);


    bool setHighSensitive(bool on);

    static int encode(const char code[], int byte);
  };
};

#endif /*! SCIP_HANDLER_H */
