/*!
  \file
  \brief SCIP プロトコル処理

  \author Satofumi KAMIMURA

  $Id: ScipHandler.cpp 212 2008-09-04 03:48:22Z satofumi $

  \todo 取得範囲以外の領域には、0 を埋める (特に、データの先頭部分)
  \todo SCIP1.1 しか使えないことを検出してエラーメッセージを出力させる
*/

#include "ScipHandler.h"
#include "Connection.h"
#include "ConnectionUtils.h"
#include "SensorParameter.h"

using namespace qrk;


struct ScipHandler::pImpl
{
  enum {
    ChecksumError = ErrorLastIndex - 1,
    PacketInvalid = ErrorLastIndex - 2,
    NotConnected = ErrorLastIndex - 3,
  };

  enum {
    Timeout = 1000,
    EachTimeotu = 10,
    LineMax = 64 + 1 + 1,
  };

  std::string error_message_;
  Connection* con_;
  bool laser_output_;


  pImpl(void)
    : error_message_("no error."), con_(NULL), laser_output_(StateUnknown) {
  }


  bool isValidConnection(void)
  {
    if (! con_) {
      error_message_ = "Connection object is not registered.";
      return false;
    } else {
      return true;
    }
  }


  int recvReply(int timeout, std::vector<std::string>* lines = NULL)
  {
    // エコーバックの読み出し
    char buffer[LineMax];
    int n = readline(con_, buffer, LineMax, timeout);
    if (n <= 0) {
      return RecvTimeout;
    }
    // !!! 送ったコマンドとエコーバックを比較できるようにするべき

    // 応答の読み出し
    n = readline(con_, buffer, LineMax, timeout);
    if (n <= 0) {
      return RecvTimeout;
    }

    // ステータスの取得
    enum { SCIP20, SCIP11 };
    int type = SCIP20;
    if (n == 1) {
      type = SCIP11;
    }
    int reply_code = -1;
    std::string reply_line = buffer;
    if (! getErrorCode(reply_line, &reply_code)) {
      return ChecksumError;
    }

    // 改行(1 byte)の読み捨て
    char last_ch;
    n = con_->recv(&last_ch, 1, timeout);
    if (n != 1) {
      return PacketInvalid;
    }

    if (! qrk::isLF(last_ch)) {
      // 文字が改行でなければ、データの可能性があるため書き戻す
      con_->ungetc(last_ch);
    }

    // lines が NULL でなければ、最初に改行が来るまでを文字列データとみなす
    if (lines != NULL) {
      while ((n = readline(con_, buffer, LineMax, timeout)) > 0) {
        lines->push_back(buffer);
        if ((n == 1) && qrk::isLF(buffer[0])) {
          // パケット最後の改行(先頭文字が改行)を検出したら、抜ける
          break;
        }
      }
    }

    // SCIP1.1 応答のときには、負号を付けて返す
    // つまり、SCIP1.1 のときは、応答がゼロ以外ならば、エラー扱いとなる
    return reply_code * ((type == SCIP11) ? -1 : +1);
  }


  char calculateChecksum(const char* buf, size_t count)
  {
    char sum = 0;
    for (size_t i = 0; i < count; ++i) {
      sum += buf[i];
    }
    return sum;
  }


  bool checkLineChecksum(std::string& line)
  {
    size_t n = line.size();
    char checksum = (calculateChecksum(&line[0], n - 1) & 0x3f) + 0x30;
    return (line[n - 1] != checksum) ? false : true;
  }


  bool getErrorCode(std::string& line, int* error_code)
  {
    int n = line.size();
    if (n == 3) {
      // SCIP2.0 応答
      if (! checkLineChecksum(line)) {
        *error_code = ChecksumError;
        return false;
      }
      *error_code = strtol(line.substr(0, 2).c_str(), NULL, 16);
      return true;

    } else if (n == 1) {
      // SCIP1.1 応答
      *error_code = strtol(line.substr(0, 1).c_str(), NULL, 16);
      return true;

    } else {
      // 不明なパケット形式
      *error_code = PacketInvalid;
      return false;
    }
  }


  bool getTimestamp(std::string& line, long* timestamp)
  {
    if (! checkLineChecksum(line)) {
      *timestamp = ChecksumError;
      return false;
    }

    *timestamp = encode(&line[0], 4);
    return true;
  }


  int substr2int(std::string& line, int from_n,
                 int length = std::string::npos)
  {
    return atoi(line.substr(from_n, length).c_str());
  }


  bool recvReplyLines(const char* message, size_t n,
                      std::vector<std::string>& lines)
  {
    con_->send(message, n);
    int ret = recvReply(Timeout, &lines);
    if (ret != 0x00) {
      // ASCII 文字列の長さを求める
      int length = 0;
      while (isprint(message[length])) {
        ++length;
      }
      char buffer[12];
      snprintf(buffer, 12, "%d", ret);
      error_message_ =
        std::string(message).substr(0, length) + " fail: " + buffer;
      return false;
    }
    return true;
  }

  CaptureType identifyPacketType(std::string& line) {

    typedef struct {
      const char* tag;
      CaptureType type;
    } tag_types_t;
    const tag_types_t compare_tags[] = {
      { "MD", MD },
      { "MS", MS },
      { "GD", GD },
      { "GS", GS },
      { "ME", ME },
      { "QT", QT },
      { "G", Classic_G },
      { NULL, TypeUnknown },
    };

    CaptureType type = TypeUnknown;
    for (size_t i = 0; i < sizeof(compare_tags)/sizeof(compare_tags[0]); ++i) {
      if (compare_tags[i].tag == NULL) {
        // 想定外のパケット
        error_message_ = "Invalid packet has arrived.";
        return type;
      }
      size_t tag_n = strlen(compare_tags[i].tag);
      if (! line.compare(0, tag_n, compare_tags[i].tag)) {
        type = compare_tags[i].type;
        break;
      }
    }
    return type;
  }


  bool recvCaptureData(RawRecvData& data)
  {
    // 応答の取得
    std::vector<std::string>& lines = data.lines;
    lines.clear();
    int n = 0;
    char buffer[LineMax];
    while ((n = readline(con_, buffer, LineMax, Timeout)) > 0) {
      lines.push_back(buffer);
      if ((n == 1) && qrk::isLF(buffer[0])) {
        // パケットの最後(先頭文字が改行)を検出したら、抜ける
        break;
      }
    }

    // 受信行の取得
    size_t lines_n = lines.size();

    // 応答タイプの判定
    if (lines_n < 2) {
      // パケットの受信行が少なすぎ
      error_message_ = "Too few receive lines";
      return false;
    }

    // パケットタイプの判定
    data.type = identifyPacketType(lines[0]);
    if (data.type == TypeUnknown) {
      return false;
    }

    if ((data.type == MD) || (data.type == MS) || (data.type == ME)) {
      return parseMdReply(data, lines);

    } else if ((data.type == GD) || (data.type == GS)) {
      return parseGdReply(data, lines);

    } else if (data.type == QT) {
      // QT のとき
      return parseQtReply(data, lines);

    } else if (data.type == Classic_G) {
      // G (SCIP 1.1) のとき
      return parseClassicGReply(data, lines);
    }

    error_message_ = "Unknown type packet is received.";
    return false;
  }


  bool parseMdReply(RawRecvData& data, std::vector<std::string>& lines)
  {
    // MD/MS 応答のパース
    if (lines[0].size() != 15) {
      error_message_ = "Invalid MD packet has arrived.";
      return false;
    }

    // エラーコードの確認
    if (! getErrorCode(lines[1], &data.error_code)) {
      error_message_ = "Invalid MD packet has arrived.";
      return false;
    }

    // ME で 'E' が返されたときは、未対応のためエラー扱いにする
    if ((data.error_code == 0xE) && (data.type == ME)) {
      error_message_ = "ME is not supported.";
      return false;
    }

    // エラーコードが 0x00 (最初の応答) のときには、正常終了の扱いにする
    if (data.error_code == 0x00) {
      data.type = Mx_Reply;
      return true;
    }

    data.capture_first = substr2int(lines[0], 2, 4);
    data.capture_last = substr2int(lines[0], 6, 4);
    data.skip_lines = substr2int(lines[0], 10, 2);
    data.skip_frames = substr2int(lines[0], 12, 1);
    data.left_times = substr2int(lines[0], 13, 2);

    // エコーバック以外の正常応答では、応答として 0x99 が返される
    if (data.error_code == 0x99) {
      // 距離データのとき
      if (! getTimestamp(lines[2], &data.timestamp)) {
        return false;
      }
    } else {
      // エコーバックのとき
      data.timestamp = -1;
    }
    return true;
  }


  bool parseGdReply(RawRecvData& data, std::vector<std::string>& lines)
  {
    // GD/GS 応答のパース
    if (lines[0].size() != 12) {
      error_message_ = "Invalid GD packet has arrived.";
      return false;
    }

    // エラーコードの確認
    if (! getErrorCode(lines[1], &data.error_code)) {
      return "Invalid GD packet has arrived.";
    }

    data.capture_first = substr2int(lines[0], 2, 4);
    data.capture_last = substr2int(lines[0], 6, 4);
    data.skip_lines = substr2int(lines[0], 10, 2);

    // GD/GS の応答では、正常時の応答は 0x00 となる
    if ((! getTimestamp(lines[2], &data.timestamp)) ||
        (data.error_code != 0x00)) {
      return false;
    }
    return true;
  }


  bool parseQtReply(RawRecvData& data, std::vector<std::string>& lines)
  {
    if (! getErrorCode(lines[1], &data.error_code)) {
      return false;
    }

    // 無効データを格納しておく
    data.timestamp = -1;
    data.capture_first = -1;
    data.capture_last = -1;
    data.skip_lines = -1;
    data.skip_frames = -1;
    data.left_times = -1;

    return true;
  }


  bool parseClassicGReply(RawRecvData& data, std::vector<std::string>& lines)
  {
    // エコーバックの確認
    if (lines[0].size() != 9) {
      error_message_ = "Invalid G Packet has arrived.";
      return false;
    }

    // エラーコードの確認
    if (! getErrorCode(lines[1], &data.error_code)) {
      return "Invalid MD packet has arrived.";
    }

    data.capture_first = substr2int(lines[0], 1, 3);
    data.capture_last = substr2int(lines[0], 4, 3);
    data.skip_lines = substr2int(lines[0], 7, 2);
    data.left_times = 0;
    data.timestamp = -1;

    return true;
  }


  std::string addLengthData(std::vector<long>& data,
                            const std::string& line,
                            const std::string& left_string,
                            const size_t data_byte, ScipVersion scip_version,
                            const int skip_lines = 1)
  {
    if (line.empty()) {
      // 空行の場合、戻る
      return left_string;
    }

    // 端数。次回に処理する分
    std::string left_byte = left_string;

    size_t data_size = (left_byte.size() + (line.size() - 1)) / data_byte;
    if (scip_version == SCIP_11) {
      // SCIP 1.1 の場合、チェックサムがないので、足し込んでおく
      data_size += 1;
    }
    size_t n = data_size * data_byte - left_byte.size();
    for (size_t i = 0; i < n; ++i) {
      left_byte.push_back(line[i]);
      if (left_byte.size() >= data_byte) {
        // データを距離に変換して、格納
        long length = encode(&left_byte[0], data_byte);
        for (int j = 0; j < skip_lines; ++j) {
          data.push_back(length);
        }
        left_byte.clear();
      }
    }
    left_byte += line.substr(n, (line.size() - n) - 1);

    return left_byte;
  }
};


ScipHandler::ScipHandler(void) : pimpl(new pImpl)
{
}


ScipHandler::~ScipHandler(void)
{
}


const char* ScipHandler::what(void)
{
  return pimpl->error_message_.c_str();
}


void ScipHandler::setConnection(Connection* con)
{
  pimpl->con_ = con;
}


Connection* ScipHandler::connection(void)
{
  return pimpl->con_;
}


int ScipHandler::sendScip20(void)
{
  if (! pimpl->isValidConnection()) {
    return pImpl::NotConnected;
  }

  pimpl->con_->send("SCIP2.0\r", 8);
  return pimpl->recvReply(pImpl::Timeout);
}


bool ScipHandler::changeBaudrate(long baudrate)
{
  if (! ((baudrate == 19200) || (baudrate == 38400) ||
         (baudrate == 57600) || (baudrate == 115200))) {
    pimpl->error_message_ = "Invalid baudrate value.";
    return false;
  }

  // SS を送信し、URG 側のボーレートを変更する
  char send_buffer[] = "SSxxxxxx\r";
  snprintf(send_buffer, 10, "SS%06ld\r", baudrate);
  pimpl->con_->send(send_buffer, 9);
  int ret = pimpl->recvReply(pImpl::Timeout);
  if (! ((ret == 0x00) || (ret == 0x03) || (ret == 0x04))) {
    pimpl->error_message_ = "Baudrate change fail.";
    return false;
  }

  // PC 側のボーレートを変更する
  if (! pimpl->con_->setBaudrate(baudrate)) {
    pimpl->error_message_ = pimpl->con_->what();
    return false;
  }
  return true;
}


bool ScipHandler::versionLines(std::vector<std::string>& lines)
{
  if (! pimpl->isValidConnection()) {
    return false;
  }

  // VV 応答の受信
  if (! pimpl->recvReplyLines("VV\r", 3, lines)) {
    return false;
  }
  return true;
}


bool ScipHandler::parameters(SensorParameter* parameter)
{
  if (! pimpl->isValidConnection()) {
    return false;
  }

  // PP 応答の受信
  std::vector<std::string> lines;
  if (! pimpl->recvReplyLines("PP\r", 3, lines)) {
    pimpl->error_message_ = "No PP response.";
    return false;
  }

  // 受信内容の格納
  int modl_length = lines[SensorParameter::MODL].size();
  if (modl_length > (5 + 2)) {
    modl_length -= (5 + 2);
  }
  parameter->model = lines[SensorParameter::MODL].substr(5, modl_length);
  parameter->distance_min = pimpl->substr2int(lines[SensorParameter::DMIN], 5);
  parameter->distance_max = pimpl->substr2int(lines[SensorParameter::DMAX], 5);
  parameter->area_total = pimpl->substr2int(lines[SensorParameter::ARES], 5);
  parameter->area_min = pimpl->substr2int(lines[SensorParameter::AMIN], 5);
  parameter->area_max = pimpl->substr2int(lines[SensorParameter::AMAX], 5);
  parameter->area_front = pimpl->substr2int(lines[SensorParameter::AFRT], 5);
  parameter->scan_rpm = pimpl->substr2int(lines[SensorParameter::SCAN], 5);

  return true;
}


int ScipHandler::setLaserOutput(LaserState output, bool force)
{
  if (! pimpl->isValidConnection()) {
    return -1;
  }

  if ((! force) && (pimpl->laser_output_ == output)) {
    // 設定と内部状態が同じだったら、戻る。ただし、強制的に設定する場合以外
    return 0;
  }

  int ret = -1;
  if (output != StateUnknown) {
    // レーザの発光状態を変更
    if (output == On) {
      pimpl->con_->send("BM\r", 3);
    } else {
      pimpl->con_->send("QT\r", 3);
    }
    ret = pimpl->recvReply(pImpl::Timeout);
  }

  // 内部状態を更新
  pimpl->laser_output_ = output;

  return ret;
}


bool ScipHandler::recvCaptureData(RawRecvData& data)
{
  return pimpl->recvCaptureData(data);
}


bool ScipHandler::convertToLength(std::vector<long>& length,
                                  const RawRecvData& data,
                                  ScipVersion scip_version)
{
  if (data.lines.size() < 4) {
    // データ行がなければ、戻る
    // !!! エラーメッセージの更新
    return false;
  }

  std::string left_string;
  size_t data_byte = ((data.type == MS) || (data.type == GS)) ? 2 : 3;

  // 最初の３行(応答、エラーコード、タイムスタンプ) は処理しない
  std::vector<std::string>::const_iterator it = data.lines.begin();
  it += (scip_version == SCIP_20) ? 3 : 2;
  for (; it != data.lines.end(); ++it) {

    // 行データを取り出し、行毎にデータの取り出しを行う
    left_string =
      pimpl->addLengthData(length, *it, left_string, data_byte, scip_version,
                           data.skip_lines);
  }
  return true;
}


bool ScipHandler::sendQT(void)
{
  int n = pimpl->con_->send("QT\r", 3);
  return (n == 3) ? true : false;
  //return pimpl->recvReply(pImpl::Timeout);
}


// 高感度計測モード
bool ScipHandler::setHighSensitive(bool on)
{
  char send_message[] = "HSx\r";
  send_message[2] = on ? '1' : '0';

  int n = pimpl->con_->send(send_message, 4);
  if (n != 4) {
    return -1;
  }
  int ret = pimpl->recvReply(pImpl::Timeout);
  if ((ret == 0x00) || (ret == 0x02)) {
    return true;
  } else {
    return true;
  }
}


int ScipHandler::encode(const char code[], int byte)
{
  int value = 0;
  for (int i = 0; i < byte; ++i) {
    value <<= 6;
    value &= ~0x3f;
    value |= code[i] - 0x30;
  }
  return value;
}
