#ifndef RANGE_SENSOR_H
#define RANGE_SENSOR_H

/*!
  \file
  \brief レンジセンサのインターフェース定義

  \author Satofumi KAMIMURA

  $Id: RangeSensor.h 171 2008-08-24 21:53:18Z satofumi $
*/

#include "stddef.h"
#include <vector>
#include <string>


namespace qrk
{
  class Connection;

  /*!
    \brief 測域センサ・インターフェース
  */
  class RangeSensor
  {
  public:
    virtual ~RangeSensor(void) {}

    /*!
      \brief 内部状態を返す

      \return 内部状態を示す文字列

      <strong>Example</strong>
\code
UrgCtrl sensor;
if (! sensor.connect(device, baudrate)) {
  printf("connect: %s\n", sensor.what());
  exit(1);
} \endcode
    */
    virtual const char* what(void) = 0;

    /*!
      \brief 接続

      \param[in] device 接続デバイス名
      \param[in] baudrate 接続ボーレート

      \retval true 接続成功
      \retval false 接続に失敗

      <strong>Example</strong>
\code
const char device[] = "/dev/ttyACM0";
const long baudrate = 115200;
UrgCtrl sensor;
if (! sensor.connect(device, baudrate)) {
  printf("connect: %s\n", sensor.what());
  exit(1);
} \endcode
    */
    virtual bool connect(const char* device, long baudrate) = 0;

    /*!
      \overload
      \brief 接続

      \param[in] con 接続オブジェクト

      \retval true 接続成功
      \retval false 接続に失敗
    */
    virtual bool connect(Connection* con) = 0;

    /*!
      \brief 接続オブジェクトの取得

      \return 接続オブジェクト
    */
    virtual Connection* connection(void) = 0;

    /*!
      \brief 切断
    */
    virtual void disconnect(void) = 0;

    /*!
      \brief 接続状態を返す

      \retval true 接続中
      \retval false 切断中
    */
    virtual bool isConnected(void) = 0;

    /*!
      \brief 有効距離の最小値を取得

      \return 有効距離の最小値
    */
    virtual long minDistance(void) const = 0;

    /*!
      \brief 有効距離の最大値を取得

      \return 有効距離の最大値
    */
    virtual long maxDistance(void) const = 0;

    /*!
      \brief １スキャンの最大データサイズを返す

      \return １スキャンの最大データサイズ
    */
    virtual int maxBufferSize(void) const = 0;

    /*!
      \brief データ取得

      データ取得を行い、バッファに格納する

      \param[out] data 取得データ用バッファ
      \param[in] max_size 取得データ用バッファの最大サイズ

      \return 取得データ数
      \retval < 0 取得失敗
    */
    virtual int capture(long data[], size_t max_size) = 0;

    virtual int intensity(long data[], size_t data_max) = 0;

    /*!
      \brief センサが返すタイムスタンプ値の設定

      !!! 説明

      \param[in] ticks その時刻のタイムスタンプの設定値
    */
    virtual bool setTimestamp(int ticks = 0) = 0;

    /*!
      \brief タイムスタンプの取得

      \attention capture() 後でなければ、この値は無効
    */
    virtual int timestamp(void) = 0;

    /*!
      \brief データインデックスのラジアン角度変換

      センサ前面が、0.0 [rad] に相当する。

      \param[in] index データインデックス

      \return ラジアン角度

      \image html sensor_radian.png センサ上面からの角度対応図
    */
    virtual double index2rad(const int index) const = 0;

    /*!
      \brief ラジアン角度のデータインデックス変換

      センサ前面が、0.0 [rad] に相当する。

      \param[in] radian ラジアン角度

      \return データインデックス

      \image html sensor_radian.png センサ上面からの角度対応図
    */
    virtual int rad2index(const double radian) const = 0;

    /*!
      \brief バージョン情報の取得

      \param[out] lines バージョン情報

      \retval true 正常終了
      \retval false 情報の取得に失敗

      <strong>Example</strong>
\code
// センサ接続
UrgCtrl sensor;
if (! sensor.connect(device)) {
  ...
}
  ...

// バージョン情報の取得
std::vector<std::string> lines;
if (sensor.versionLines(lines)) {
  for (std::vector<std::string> >::iterator it = lines.begin();
    it != lines.end(); ++it) {
    // 表示
    std::cout << *it << std::endl;
  }
} \endcode
    */
    virtual bool versionLines(std::vector<std::string>& lines) = 0;
  };
};

#endif /* !RANGE_SENSOR_H */
