/*!
  \file
  \brief 角度差の表示

  \author Satofumi KAMIMURA

  $Id: outputAngleError.cpp 223 2008-09-20 02:19:33Z satofumi $

  plot 'degree_errors.dat' using 1:2:3 with yerrorbars
*/

#include "mUrgCtrl.h"
#include "convertRangePoints.h"
#include "geometry.h"

using namespace qrk;


int main(int argc, char *argv[])
{
  mUrgCtrl urg(argc, argv);
  if (! urg.connect("/dev/ttyACM0")) {
    printf("UrgCtrl::connect: %s\n", urg.what());
    exit(1);
  }

  std::vector<long> data;
  int n = urg.capture(data);
  if (n <= 0) {
    printf("UrgCtrl::capture: %s\n", urg.what());
    exit(1);
  }

  std::vector<Grid<long> > points;
  convertRangePoints(points, data, &urg);

  if (points.empty()) {
    fprintf(stderr, "no point data.\n");
    return -1;
  }

  FILE* fd = fopen("points.dat", "w");
  if (fd) {
    for (std::vector<Grid<long> >::iterator it = points.begin();
         it != points.end(); ++it) {
      fprintf(fd, "%d\t%d\n", it->x, it->y);
    }
    fclose(fd);
  }

  const double Rad2Deg = 180.0 / M_PI;

  // 測定誤差の定義
  const long LengthError = 10;

  fd = fopen("degree_errors.dat", "w");
  if (fd) {
    int index = 0;
    Grid<long> pre_point = points[0];
    for (std::vector<Grid<long> >::iterator it = points.begin() + 1;
         it != points.end(); ++it, ++index) {
      int degree_diff = static_cast<int>(atan2(it->y - pre_point.y,
                                               it->x - pre_point.x) * Rad2Deg);

      long length = qrk::length<long>(*it, pre_point);
      long error = static_cast<int>(atan2(LengthError * 2.0, length) * Rad2Deg);

      // 測定範囲は -135 からなので、-135 より小さい角度は 180 以上として扱う
      if ((degree_diff - error) <= -135) {
        degree_diff += 135 + 180;
      }

      fprintf(fd, "%d\t%d\t%ld\n", index, degree_diff, error);

      pre_point = *it;
    }
    fclose(fd);
  }

  return 0;
}
