/*!
  \file
  \brief 線分の検出サンプル

  \author Satofumi KAMIMURA

  $Id: createLineSegmentSample.cpp 223 2008-09-20 02:19:33Z satofumi $

  plot 'points.dat', 'lines.dat' with line
  plot 'degree_errors.dat' using 1:2:3 with yerrorbars, 'degree_lines.dat' with line
*/

#include "mUrgCtrl.h"
#include "convertRangePoints.h"
#include "geometry.h"
#include "LineTypes.h"

using namespace qrk;

static FILE* DegreeLine_fd = NULL;


namespace {

  // グループを直線として登録
  // !!! 傾きと位置にメディアンの値を使った直線にすべき
  void registerLine(std::vector<Line<long> >& lines,
                    std::vector<long>& x_points,
                    std::vector<long>& y_points,
                    std::vector<double>& radians,
                    const Grid<long>& first, const Grid<long>& last, int degree,
                    int first_index, int last_index)
  {
#if 0
    long length = qrk::length<long>(first, last);

    int index = x_points.size() / 2;
    std::nth_element(x_points.begin(),
                     x_points.begin() + index, x_points.end());
    int x = x_points[index];

    std::nth_element(y_points.begin(),
                     y_points.begin() + index, y_points.end());
    int y = y_points[index];

    std::nth_element(radians.begin(),
                     radians.begin() + index, radians.end());
    double radian = radians[index];
    //double radian = degree * M_PI / 180.0;

    x = (first.x + last.x) / 2;
    y = (first.y + last.y) / 2;

    Grid<long> line_first(x - static_cast<int>(length / 2.0 * cos(radian)),
                          y - static_cast<int>(length / 2.0 * sin(radian)));
    Grid<long> line_last(x + static_cast<int>(length / 2.0 * cos(radian)),
                         y + static_cast<int>(length / 2.0 * sin(radian)));

    lines.push_back(Line<long>(line_first, line_last));
#else
    lines.push_back(Line<long>(first, last));
    double radian = atan2(first.y - last.y, first.x - last.x);
#endif

    int draw_degree = static_cast<int>(radian * 180.0 / M_PI);
    if (DegreeLine_fd) {
      fprintf(DegreeLine_fd, "%d\t%d\n%d\t%d\n\n\n",
              first_index, draw_degree, last_index, draw_degree);
    }
  }


  void detectLineSegment(std::vector<Line<long> >& lines,
                         const std::vector<Grid<long> >& points)
  {
    const double Rad2Deg = 180.0 / M_PI;
    const long LengthError = 10;

    int group_max = 0;
    int group_min = 0;
    Grid<long> line_first;
    std::vector<long> x_points;
    std::vector<long> y_points;
    std::vector<double> radians;

    FILE* fd = fopen("degree_errors.dat", "w");

    DegreeLine_fd = fopen("degree_lines.dat", "w");

    int index = 0;

    int first_index = 0;

    Grid<long> pre_point = points[0];
    for (std::vector<Grid<long> >::const_iterator it = points.begin() + 1;
         it != points.end(); ++it, ++index) {
      int degree_diff = static_cast<int>(atan2(it->y - pre_point.y,
                                               it->x - pre_point.x) * Rad2Deg);

      long length = qrk::length<long>(*it, pre_point);
      long error = static_cast<int>(atan2(LengthError * 2.0, length) * Rad2Deg);

      // 測定範囲は -135 からなので、-135 より小さい角度は 180 以上として扱う
      if ((degree_diff - error) <= -135) {
        degree_diff += 135 + 180;
      }

      if (fd) {
        fprintf(fd, "%d\t%d\t%ld\n", index, degree_diff, error);
      }

      int current_max = degree_diff + error;
      int current_min = degree_diff - error;

      if (x_points.empty()) {
        // 始点の登録
        line_first = pre_point;
        group_max = current_max;
        group_min = current_min;
        first_index = index;

        // !!! この追加処理は、なんとかして削除すべき。下の実装と重複している
        x_points.push_back(it->x);
        y_points.push_back(it->y);
        radians.push_back(atan2(it->y - pre_point.y, it->x - pre_point.x));

      } else {
        if ((group_max >= current_min) && (group_min <= current_max)) {
          // グループに点を登録
          if (current_max < group_max) {
            group_max = current_max;
          }
          if (current_min > group_min) {
            group_min = current_min;
          }
          x_points.push_back(it->x);
          y_points.push_back(it->y);
          // !!! 重複している
          radians.push_back(atan2(it->y - pre_point.y, it->x - pre_point.x));

        } else {
          if (x_points.size() >= 5) {
            // グループを直線として登録
            registerLine(lines, x_points, y_points, radians,
                         line_first, pre_point, (group_max + group_min) / 2,
                         first_index, index);
          }
          x_points.clear();
          y_points.clear();
          radians.clear();
        }
      }
      pre_point = *it;
    }

    // !!! くくること
    if (x_points.size() >= 5) {
      registerLine(lines, x_points, y_points,
                   radians, line_first, points.back(),
                   (group_max + group_min) / 2,
                   first_index, index);
    }

    if (fd) {
      fclose(fd);
    }
    if (DegreeLine_fd) {
      fclose(DegreeLine_fd);
    }
  }
};


int main(int argc, char *argv[])
{
  mUrgCtrl urg(argc, argv);
  if (! urg.connect("/dev/ttyACM0")) {
    printf("UrgCtrl::connect: %s\n", urg.what());
    exit(1);
  }

  std::vector<long> data;
  int n = urg.capture(data);
  if (n <= 0) {
    printf("UrgCtrl::capture: %s\n", urg.what());
    exit(1);
  }

  std::vector<Grid<long> > points;
  convertRangePoints(points, data, &urg);

  if (points.empty()) {
    fprintf(stderr, "no point data.\n");
    return -1;
  }

  FILE* fd = fopen("points.dat", "w");
  if (fd) {
    for (std::vector<Grid<long> >::iterator it = points.begin();
         it != points.end(); ++it) {
      fprintf(fd, "%d\t%d\n", it->x, it->y);
    }
    fclose(fd);
  }

  // 直線検出
  std::vector<Line<long> > lines;
  detectLineSegment(lines, points);

  fd = fopen("lines.dat", "w");
  if (fd) {
    for (std::vector<Line<long> >::iterator it = lines.begin();
         it != lines.end(); ++it) {
      fprintf(fd, "%d\t%d\n%d\t%d\n",
              it->start.x, it->start.y, it->end.x, it->end.y);
      fprintf(fd, "\n\n");
    }
    fclose(fd);
  }

  return 0;
}
